<?php

namespace App\Console\Commands;

use App\Services\Inventory\ForecastingService;
use Illuminate\Console\Command;

class ComputeInventoryVelocity extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:compute-velocity 
                            {--dish-id= : Compute velocity for specific dish ID}
                            {--force : Force update even if recently updated}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Compute 7-day and 30-day velocity for dishes based on sales movements';

    protected ForecastingService $forecastingService;

    /**
     * Create a new command instance.
     */
    public function __construct(ForecastingService $forecastingService)
    {
        parent::__construct();
        $this->forecastingService = $forecastingService;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('🚀 Starting inventory velocity computation...');
        $startTime = microtime(true);

        try {
            $dishId = $this->option('dish-id');
            
            if ($dishId) {
                $this->computeSpecificDish($dishId);
            } else {
                $this->computeAllDishes();
            }

            $executionTime = round((microtime(true) - $startTime) * 1000, 2);
            $this->info("✅ Velocity computation completed in {$executionTime}ms");
            
            return Command::SUCCESS;
            
        } catch (\Exception $e) {
            $this->error("❌ Velocity computation failed: " . $e->getMessage());
            $this->error($e->getTraceAsString());
            
            return Command::FAILURE;
        }
    }

    /**
     * Compute velocity for a specific dish
     */
    private function computeSpecificDish(int $dishId): void
    {
        $dish = \App\Models\Dish::find($dishId);
        
        if (!$dish) {
            $this->error("❌ Dish with ID {$dishId} not found");
            return;
        }

        $this->info("📊 Computing velocity for dish: {$dish->name} (ID: {$dishId})");
        
        $this->forecastingService->updateDishVelocity($dish);
        
        $dish->refresh();
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['7-day velocity', "{$dish->velocity_7d} units/day"],
                ['30-day velocity', "{$dish->velocity_30d} units/day"],
                ['Current stock', $dish->stock],
                ['Projected runway', "{$dish->projected_runway_days} days"],
                ['Last updated', $dish->velocity_updated_at?->format('Y-m-d H:i:s') ?? 'Never'],
            ]
        );
    }

    /**
     * Compute velocity for all dishes
     */
    private function computeAllDishes(): void
    {
        $this->info('📊 Computing velocity for all dishes...');
        
        $results = $this->forecastingService->updateVelocityForAllDishes();
        
        $this->table(
            ['Metric', 'Count'],
            [
                ['Total processed', $results['processed']],
                ['Successfully updated', $results['updated']],
                ['Errors encountered', $results['errors']],
            ]
        );

        if ($results['errors'] > 0) {
            $this->warn("⚠️  {$results['errors']} dishes failed to update. Check logs for details.");
        }

        // Show velocity statistics
        $this->info('📈 Velocity Statistics:');
        $stats = $this->forecastingService->getVelocityStatistics();
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['Total dishes', $stats['total_dishes']],
                ['Dishes with sales', $stats['dishes_with_sales']],
                ['Dishes without sales', $stats['dishes_without_sales']],
                ['Average 7-day velocity', "{$stats['average_velocity_7d']} units/day"],
                ['Average 30-day velocity', "{$stats['average_velocity_30d']} units/day"],
                ['Maximum 7-day velocity', "{$stats['max_velocity_7d']} units/day"],
                ['Minimum 7-day velocity', "{$stats['min_velocity_7d']} units/day"],
            ]
        );
    }
}
