<?php

namespace App\Console\Commands;

use App\Services\Inventory\PerformanceService;
use Carbon\Carbon;
use Illuminate\Console\Command;

class CreateInventorySnapshots extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:create-snapshots 
                            {--date= : Create snapshots for specific date (YYYY-MM-DD)}
                            {--range= : Create snapshots for date range in days (e.g., 7 for last 7 days)}
                            {--force : Recreate snapshots even if they already exist}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create daily inventory snapshots for historical KPI tracking and analysis';

    protected PerformanceService $performanceService;

    /**
     * Create a new command instance.
     */
    public function __construct(PerformanceService $performanceService)
    {
        parent::__construct();
        $this->performanceService = $performanceService;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('📸 Starting inventory snapshot creation...');
        $startTime = microtime(true);

        try {
            $date = $this->option('date');
            $range = $this->option('range');

            if ($range) {
                $this->createRangeSnapshots((int) $range);
            } elseif ($date) {
                $this->createDateSnapshot(Carbon::parse($date));
            } else {
                $this->createDefaultSnapshot();
            }

            $executionTime = round((microtime(true) - $startTime) * 1000, 2);
            $this->info("✅ Snapshot creation completed in {$executionTime}ms");
            
            return Command::SUCCESS;
            
        } catch (\Exception $e) {
            $this->error("❌ Snapshot creation failed: " . $e->getMessage());
            return Command::FAILURE;
        }
    }

    /**
     * Create snapshots for a date range
     */
    private function createRangeSnapshots(int $days): void
    {
        $this->info("📅 Creating snapshots for last {$days} days...");
        
        $results = [
            'total_days' => $days,
            'successful_days' => 0,
            'skipped_days' => 0,
            'error_days' => 0,
        ];

        for ($i = 1; $i <= $days; $i++) {
            $date = Carbon::now()->subDays($i);
            
            try {
                // Check if snapshots already exist
                if (!$this->option('force') && $this->snapshotsExist($date)) {
                    $this->line("⏭️  Skipping {$date->format('Y-m-d')} - snapshots already exist");
                    $results['skipped_days']++;
                    continue;
                }

                $dayResults = $this->performanceService->createDailySnapshots($date);
                $this->line("✅ Created snapshots for {$date->format('Y-m-d')}: {$dayResults['dish_snapshots']} dishes");
                $results['successful_days']++;
                
            } catch (\Exception $e) {
                $this->error("❌ Failed to create snapshots for {$date->format('Y-m-d')}: " . $e->getMessage());
                $results['error_days']++;
            }
        }

        $this->displayRangeResults($results);
    }

    /**
     * Create snapshot for a specific date
     */
    private function createDateSnapshot(Carbon $date): void
    {
        $this->info("📅 Creating snapshots for {$date->format('Y-m-d')}...");
        
        if (!$this->option('force') && $this->snapshotsExist($date)) {
            if (!$this->confirm("Snapshots already exist for {$date->format('Y-m-d')}. Recreate them?")) {
                $this->info('Snapshot creation cancelled.');
                return;
            }
            
            // Delete existing snapshots
            $this->deleteExistingSnapshots($date);
        }

        $results = $this->performanceService->createDailySnapshots($date);
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['Date', $results['date']],
                ['Dish snapshots created', $results['dish_snapshots']],
                ['Global snapshot created', $results['global_snapshot'] ? 'Yes' : 'No'],
                ['Errors encountered', $results['errors']],
            ]
        );

        if ($results['errors'] > 0) {
            $this->warn("⚠️  {$results['errors']} errors occurred. Check logs for details.");
        }
    }

    /**
     * Create default snapshot (yesterday)
     */
    private function createDefaultSnapshot(): void
    {
        $date = Carbon::yesterday();
        $this->info("📅 Creating default snapshots for yesterday ({$date->format('Y-m-d')})...");
        
        $results = $this->performanceService->createDailySnapshots($date);
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['Date', $results['date']],
                ['Dish snapshots created', $results['dish_snapshots']],
                ['Global snapshot created', $results['global_snapshot'] ? 'Yes' : 'No'],
                ['Errors encountered', $results['errors']],
            ]
        );

        // Show some snapshot statistics
        $this->displaySnapshotStatistics($date);
    }

    /**
     * Display range creation results
     */
    private function displayRangeResults(array $results): void
    {
        $this->table(
            ['Metric', 'Count'],
            [
                ['Total days requested', $results['total_days']],
                ['Successfully created', $results['successful_days']],
                ['Skipped (already exist)', $results['skipped_days']],
                ['Failed with errors', $results['error_days']],
            ]
        );

        if ($results['error_days'] > 0) {
            $this->warn("⚠️  {$results['error_days']} days failed. Check logs for details.");
        }

        if ($results['successful_days'] > 0) {
            $this->info("✨ Successfully created snapshots for {$results['successful_days']} days.");
        }
    }

    /**
     * Display snapshot statistics for a date
     */
    private function displaySnapshotStatistics(Carbon $date): void
    {
        $snapshots = \App\Models\InventorySnapshot::where('snapshot_date', $date->format('Y-m-d'))
            ->dishSnapshots()
            ->get();

        if ($snapshots->isEmpty()) {
            return;
        }

        $this->info('📊 Snapshot Statistics:');
        
        $stats = [
            'total_dishes' => $snapshots->count(),
            'dishes_with_sales' => $snapshots->where('sales_qty', '>', 0)->count(),
            'total_sales_qty' => $snapshots->sum('sales_qty'),
            'total_sales_value' => $snapshots->sum('sales_value'),
            'out_of_stock_events' => $snapshots->where('was_out_of_stock', true)->count(),
            'low_stock_events' => $snapshots->where('was_low_stock', true)->count(),
            'avg_velocity' => $snapshots->avg('velocity_7d'),
        ];

        $this->table(
            ['Metric', 'Value'],
            [
                ['Total dishes tracked', $stats['total_dishes']],
                ['Dishes with sales', $stats['dishes_with_sales']],
                ['Total sales quantity', number_format($stats['total_sales_qty'])],
                ['Total sales value', "₦" . number_format($stats['total_sales_value'], 2)],
                ['Out of stock events', $stats['out_of_stock_events']],
                ['Low stock events', $stats['low_stock_events']],
                ['Average velocity', round($stats['avg_velocity'], 2) . " units/day"],
            ]
        );
    }

    /**
     * Check if snapshots already exist for a date
     */
    private function snapshotsExist(Carbon $date): bool
    {
        return \App\Models\InventorySnapshot::where('snapshot_date', $date->format('Y-m-d'))->exists();
    }

    /**
     * Delete existing snapshots for a date
     */
    private function deleteExistingSnapshots(Carbon $date): void
    {
        $deleted = \App\Models\InventorySnapshot::where('snapshot_date', $date->format('Y-m-d'))->delete();
        $this->info("🗑️  Deleted {$deleted} existing snapshots for {$date->format('Y-m-d')}");
    }
}
