<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

class GenerateDishManifest extends Command
{
    protected $signature = 'images:manifest-dishes {--rebuild : Re-scan storage/app/public/dishes and rebuild manifest entries for missing dishes}';
    protected $description = 'Rebuild or update the public/assets/image-manifest.json with entries for existing dish originals (JPEG variants only).';

    public function handle(): int
    {
        $storage = Storage::disk('public');
        $dir = 'dishes';
        if (!$storage->exists($dir)) {
            $this->warn('No dishes directory found in public disk.');
            return Command::SUCCESS;
        }
        $files = collect($storage->files($dir))
            ->filter(fn($f)=>preg_match('/\.(jpe?g|png)$/i',$f));
        if ($files->isEmpty()) {
            $this->info('No dish originals found.');
            return Command::SUCCESS;
        }
        $manifestFile = public_path('assets/image-manifest.json');
        $manifest = [ 'generatedAt' => now()->toIso8601String(), 'slider' => [], 'dishes' => [] ];
        if (is_file($manifestFile)) {
            try { $decoded = json_decode(file_get_contents($manifestFile), true); if (is_array($decoded)) $manifest = array_merge($manifest, $decoded); } catch (\Throwable $e) {}
        }
        $updated = 0;
        foreach ($files as $path) {
            $full = $storage->path($path);
            $baseName = pathinfo($full, PATHINFO_FILENAME);
            // Skip if already present and no rebuild requested.
            if (!$this->option('rebuild')) {
                $exists = collect($manifest['dishes'] ?? [])->firstWhere('base', $baseName);
                if ($exists) continue;
            }
            // Add minimal entry pointing to original so component can still render.
            $manifest['dishes'] = array_values(array_filter($manifest['dishes'], fn($e)=>($e['base']??null) !== $baseName));
            $manifest['dishes'][] = [
                'type' => 'dish',
                'base' => $baseName,
                'variants' => [],
                'thumb' => null,
                'lqip' => null,
                'original' => 'storage/'.$path,
                'partial' => true,
            ];
            $updated++;
        }
        $manifest['generatedAt'] = now()->toIso8601String();
        if (!is_dir(dirname($manifestFile))) @mkdir(dirname($manifestFile), 0775, true);
        file_put_contents($manifestFile, json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
        $this->info("Updated manifest with $updated dish entries.");
        return Command::SUCCESS;
    }
}
