<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class GenerateIcons extends Command
{
    protected $signature = 'icons:generate 
        {--source= : Source logo path relative to public, default assets/logo.png}
        {--bg=#4f46e5 : Background color hex for maskable icons (e.g. #4f46e5); use transparent for non-maskable}
        {--padding=0.82 : Content scale inside canvas for maskable icons (0.6 - 0.95)}
        {--mono : Generate additional monochrome maskable icons (white glyph) }';
    protected $description = 'Generate favicons and PWA app icons from logo using GD (brand background, padding, maskable, optional monochrome)';
    protected $aliases = [
        'icons:gen',      // short
        'pwa:icons',      // semantic alias
        'pwa:icons:generate', // verbose alias
    ];

    public function handle(): int
    {
        $publicPath = public_path();
        $srcRel = $this->option('source') ?: 'assets/logo.png';
        $src = $publicPath . DIRECTORY_SEPARATOR . str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $srcRel);
        $outDir = $publicPath . DIRECTORY_SEPARATOR . 'icons';

        if (!file_exists($src)) {
            $this->error("Logo not found at {$srcRel}");
            return self::FAILURE;
        }

        if (!extension_loaded('gd')) {
            $this->error('PHP GD extension is required. Enable php_gd in your php.ini');
            return self::FAILURE;
        }

        if (!is_dir($outDir)) {
            mkdir($outDir, 0775, true);
        }

        $outputs = [
            ['name' => 'favicon-16x16.png', 'size' => 16],
            ['name' => 'favicon-32x32.png', 'size' => 32],
            ['name' => 'apple-touch-icon.png', 'size' => 180],
            ['name' => 'android-chrome-192x192.png', 'size' => 192],
            ['name' => 'android-chrome-256x256.png', 'size' => 256],
            ['name' => 'android-chrome-384x384.png', 'size' => 384],
            ['name' => 'android-chrome-512x512.png', 'size' => 512],
            ['name' => 'maskable-192x192.png', 'size' => 192, 'maskable' => true],
            ['name' => 'maskable-512x512.png', 'size' => 512, 'maskable' => true],
        ];

        $bgHex = (string) $this->option('bg');
        $padding = max(0.6, min(0.95, (float) $this->option('padding')));
        $mono = (bool) $this->option('mono');

        foreach ($outputs as $o) {
            $dest = $outDir . DIRECTORY_SEPARATOR . $o['name'];
            $this->generateIcon($src, $dest, $o['size'], !empty($o['maskable']), $bgHex, $padding, false);
            $this->info("Generated icons/{$o['name']}");
        }

        if ($mono) {
            foreach ([192, 512] as $size) {
                $dest = $outDir . DIRECTORY_SEPARATOR . "maskable-mono-{$size}x{$size}.png";
                $this->generateIcon($src, $dest, $size, true, $bgHex, $padding, true);
                $this->info("Generated icons/" . basename($dest));
            }
        }

        $this->info('All icons generated.');
        return self::SUCCESS;
    }

    /**
     * Resize with contain fit into a square canvas, keep transparency.
     */
    protected function resizeContainPng(string $srcPath, string $destPath, int $size, bool $maskable = false, ?array $bgRgb = null, float $paddingFactor = 0.82, bool $monochrome = false): void
    {
        $src = imagecreatefrompng($srcPath);
        if (!$src) {
            throw new \RuntimeException('Unable to read PNG: ' . $srcPath);
        }
        $srcW = imagesx($src);
        $srcH = imagesy($src);

        // Use provided padding for maskable, else full-size
        $paddingFactor = $maskable ? $paddingFactor : 1.0;
        $contentSize = (int) floor($size * $paddingFactor);

        $ratio = min($contentSize / $srcW, $contentSize / $srcH);
        $dstW = max(1, (int) floor($srcW * $ratio));
        $dstH = max(1, (int) floor($srcH * $ratio));

        $canvas = imagecreatetruecolor($size, $size);
        imagealphablending($canvas, false);
        imagesavealpha($canvas, true);

        if ($maskable && $bgRgb) {
            $bg = imagecolorallocate($canvas, $bgRgb['r'], $bgRgb['g'], $bgRgb['b']);
            imagefilledrectangle($canvas, 0, 0, $size, $size, $bg);
        } else {
            $transparent = imagecolorallocatealpha($canvas, 0, 0, 0, 127);
            imagefilledrectangle($canvas, 0, 0, $size, $size, $transparent);
        }

        $dstX = (int) floor(($size - $dstW) / 2);
        $dstY = (int) floor(($size - $dstH) / 2);

        if ($monochrome) {
            $src = $this->toMonochromeGlyph($src);
            $srcW = imagesx($src);
            $srcH = imagesy($src);
        }

        imagecopyresampled($canvas, $src, $dstX, $dstY, 0, 0, $dstW, $dstH, $srcW, $srcH);

        imagepng($canvas, $destPath, 9);

        imagedestroy($src);
        imagedestroy($canvas);
    }

    protected function hexToRgb(string $hex): array
    {
        $hex = ltrim(trim($hex), '#');
        if (strlen($hex) === 3) {
            $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        }
        $int = hexdec($hex);
        return [
            'r' => ($int >> 16) & 255,
            'g' => ($int >> 8) & 255,
            'b' => $int & 255,
        ];
    }

    protected function generateIcon(string $src, string $dest, int $size, bool $maskable, string $bgHex, float $padding, bool $monochrome): void
    {
        $rgb = $maskable ? $this->hexToRgb($bgHex) : null;
        $this->resizeContainPng($src, $dest, $size, $maskable, $rgb, $padding, $monochrome);
    }

    protected function toMonochromeGlyph($src)
    {
        $w = imagesx($src);
        $h = imagesy($src);
        $glyph = imagecreatetruecolor($w, $h);
        imagealphablending($glyph, false);
        imagesavealpha($glyph, true);
        $transparent = imagecolorallocatealpha($glyph, 0, 0, 0, 127);
        imagefilledrectangle($glyph, 0, 0, $w, $h, $transparent);
        for ($y = 0; $y < $h; $y++) {
            for ($x = 0; $x < $w; $x++) {
                $rgba = imagecolorat($src, $x, $y);
                $a = ($rgba & 0x7F000000) >> 24; // 0 opaque, 127 transparent
                if ($a < 127) {
                    $col = imagecolorallocatealpha($glyph, 255, 255, 255, $a);
                    imagesetpixel($glyph, $x, $y, $col);
                }
            }
        }
        imagedestroy($src);
        return $glyph;
    }
}
