<?php

namespace App\Console\Commands;

use App\Services\Inventory\ForecastingService;
use Illuminate\Console\Command;

class GenerateInventoryAlerts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:generate-alerts 
                            {--dish-id= : Generate alerts for specific dish ID}
                            {--type= : Generate specific alert type (low_stock, out_of_stock, etc.)}
                            {--dry-run : Show what alerts would be generated without creating them}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate inventory alerts based on stock levels and velocity predictions';

    protected ForecastingService $forecastingService;

    /**
     * Create a new command instance.
     */
    public function __construct(ForecastingService $forecastingService)
    {
        parent::__construct();
        $this->forecastingService = $forecastingService;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('🚨 Starting inventory alert generation...');
        $startTime = microtime(true);

        try {
            if ($this->option('dry-run')) {
                $this->info('🔍 DRY RUN MODE - No alerts will be created');
            }

            $results = $this->forecastingService->generateAlerts();

            $this->displayResults($results);

            // Show alert summary
            $this->displayAlertSummary();

            $executionTime = round((microtime(true) - $startTime) * 1000, 2);
            $this->info("✅ Alert generation completed in {$executionTime}ms");
            
            return Command::SUCCESS;
            
        } catch (\Exception $e) {
            $this->error("❌ Alert generation failed: " . $e->getMessage());
            $this->error($e->getTraceAsString());
            
            return Command::FAILURE;
        }
    }

    /**
     * Display generation results
     */
    private function displayResults(array $results): void
    {
        $this->table(
            ['Metric', 'Count'],
            [
                ['Dishes processed', $results['processed']],
                ['New alerts created', $results['alerts_created']],
                ['Alerts resolved', $results['alerts_resolved']],
            ]
        );

        if ($results['alerts_created'] > 0) {
            $this->info("🆕 {$results['alerts_created']} new alerts created");
        }

        if ($results['alerts_resolved'] > 0) {
            $this->info("✅ {$results['alerts_resolved']} alerts automatically resolved");
        }

        if ($results['alerts_created'] === 0 && $results['alerts_resolved'] === 0) {
            $this->info("😌 No new alerts generated - all inventory levels are acceptable");
        }
    }

    /**
     * Display current alert summary
     */
    private function displayAlertSummary(): void
    {
        $summary = $this->forecastingService->getCriticalAlertsSummary();

        if ($summary['total_active'] === 0) {
            $this->info("✨ No active alerts - inventory is in good shape!");
            return;
        }

        $this->warn("📊 Active Alert Summary:");
        
        $this->table(
            ['Severity', 'Count'],
            [
                ['🔴 Critical', $summary['critical']],
                ['🟠 High', $summary['high']],
                ['🟡 Medium', $summary['medium']],
                ['🟢 Low', $summary['low']],
                ['📋 Total', $summary['total_active']],
            ]
        );

        $this->table(
            ['Alert Type', 'Count'],
            [
                ['Out of Stock', $summary['by_type']['out_of_stock']],
                ['Low Stock', $summary['by_type']['low_stock']],
                ['Projected Stockout', $summary['by_type']['projected_stockout']],
                ['Safety Stock Breach', $summary['by_type']['safety_stock_breach']],
            ]
        );

        // Show most critical alerts
        $criticalAlerts = \App\Models\InventoryAlert::active()
            ->with('dish:id,name')
            ->orderBy('severity', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        if ($criticalAlerts->isNotEmpty()) {
            $this->warn("🚨 Most Critical Alerts:");
            
            $alertsData = $criticalAlerts->map(function ($alert) {
                return [
                    $alert->getSeverityBadge(),
                    $alert->getTypeLabel(),
                    $alert->dish->name ?? 'Unknown',
                    $alert->title,
                    $alert->created_at->diffForHumans(),
                ];
            })->toArray();

            $this->table(
                ['Severity', 'Type', 'Dish', 'Title', 'Created'],
                $alertsData
            );
        }

        if ($summary['critical'] > 0) {
            $this->error("⚠️  {$summary['critical']} CRITICAL alerts require immediate attention!");
        }
    }
}
