<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use App\Models\Dish;

class ImagesConsolidate extends Command
{
    protected $signature = 'images:consolidate {--force : Overwrite existing originals when names collide} {--dry : Dry-run only}';
    protected $description = 'Consolidate scattered dish/slider images into a single storage disk path and rewrite DB references.';

    public function handle(): int
    {
        $diskName = config('images.disk','public');
        $disk = Storage::disk($diskName);
        $dishesDir = trim(config('images.paths.dishes','dishes'),'/');

        $dry = $this->option('dry');
        $force = $this->option('force');
        $updated = 0; $skipped = 0; $copied = 0; $errors = 0;

        Dish::chunkById(100, function($chunk) use (&$updated,&$skipped,&$copied,$disk,$dishesDir,$dry,$force,&$errors){
            foreach ($chunk as $dish) {
                $orig = $dish->image; if (!$orig) { $skipped++; continue; }
                // Already consolidated? (stored relative path like dishes/<hash>.jpg)
                if (!str_starts_with($orig,'http') && preg_match('#^'.preg_quote($dishesDir,'#').'/#',$orig)) { $skipped++; continue; }

                // Resolve current public file path
                $publicCandidate = null;
                if (str_starts_with($orig,'assets/')) {
                    $publicCandidate = public_path($orig);
                } elseif (is_file(public_path($orig))) {
                    $publicCandidate = public_path($orig);
                } elseif (is_file(public_path('storage/'.ltrim($orig,'/')))) {
                    $publicCandidate = public_path('storage/'.ltrim($orig,'/'));
                }
                if (!$publicCandidate || !is_file($publicCandidate)) {
                    $this->warn("Missing source for dish #{$dish->id}: {$orig}");
                    $errors++; continue;
                }

                $ext = strtolower(pathinfo($publicCandidate, PATHINFO_EXTENSION)) ?: 'jpg';
                // Generate stable hash name based on original file contents
                $hash = substr(sha1_file($publicCandidate),0,24);
                $targetRel = $dishesDir.'/'.$hash.'.'.$ext;
                if (!$disk->exists($targetRel) || $force) {
                    if (!$dry) {
                        $disk->put($targetRel, file_get_contents($publicCandidate));
                    }
                    $copied++;
                }

                if ($dish->image !== $targetRel) {
                    if (!$dry) {
                        $dish->image = $targetRel; // relative path stored
                        $dish->saveQuietly();
                    }
                    $updated++;
                }
            }
        });

        $this->line("Consolidation complete. updated={$updated} copied={$copied} skipped={$skipped} errors={$errors} dry=".($dry?'yes':'no'));
        if ($dry) $this->info('Dry-run only. Re-run without --dry to apply.');
        $this->info("Example consolidated path pattern: {$dishesDir}/<hash>.jpg (served via /storage/ after storage:link)");
        return self::SUCCESS;
    }
}
