<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager; // we'll implement using GD built-in manually (no dependency add now)
use App\Models\Dish;

class ImagesGenerateVariants extends Command
{
    protected $signature = 'images:variants {--force : Rebuild variants even if they exist} {--type=dishes : dishes|sliders|all}';
    protected $description = 'Generate responsive image variants and rebuild the image manifest.';

    public function handle(): int
    {
        $typeOpt = $this->option('type');
        $types = $typeOpt === 'all' ? ['dishes','sliders'] : [$typeOpt];
        $force = $this->option('force');
        $disk = Storage::disk(config('images.disk','public'));
        $manifest = [ 'generatedAt' => now()->toIso8601String(), 'slider' => [], 'dishes' => [] ];

        foreach ($types as $group) {
            if ($group === 'dishes') {
                $this->generateDishVariants($disk, $manifest, $force);
            } elseif ($group === 'sliders') {
                $this->generateSliderVariants($disk, $manifest, $force);
            }
        }

        file_put_contents(public_path('assets/image-manifest.json'), json_encode($manifest, JSON_PRETTY_PRINT));
        $this->info('Manifest rebuilt: assets/image-manifest.json');
        return self::SUCCESS;
    }

    protected function generateDishVariants($disk, array &$manifest, bool $force): void
    {
        $variantsWidths = config('images.variants.dishes', [200,400,600]);
        $basePublic = public_path('assets/dishes');
        if (!is_dir($basePublic)) mkdir($basePublic, 0775, true);
        $count = 0;
        Dish::chunkById(100, function($chunk) use (&$manifest,$variantsWidths,$basePublic,$disk,$force,&$count){
            foreach ($chunk as $dish) {
                $rel = $dish->image; if (!$rel) continue;
                if (!preg_match('#^dishes/[a-f0-9]{20,}#',$rel)) continue; // only consolidated hashed names
                $originalPath = $disk->path($rel);
                if (!is_file($originalPath)) continue;
                $hashBase = pathinfo($originalPath, PATHINFO_FILENAME);
                $ext = strtolower(pathinfo($originalPath, PATHINFO_EXTENSION));
                $variantsMeta = [];
                foreach ($variantsWidths as $w) {
                    $destRel = "assets/dishes/{$hashBase}-{$w}.jpg";
                    $destAbs = public_path($destRel);
                    if (is_file($destAbs) && !$force) {
                        [$width] = getimagesize($destAbs) ?: [null];
                        if ($width) $variantsMeta[] = ['width'=>$width,'jpg'=>$destRel];
                        continue;
                    }
                    try {
                        $this->resizeToWidth($originalPath, $destAbs, $w);
                        [$width] = getimagesize($destAbs) ?: [$w];
                        $variantsMeta[] = ['width'=>$width,'jpg'=>$destRel];
                    } catch (\Throwable $e) {
                        $this->warn('Variant gen failed: '.$e->getMessage());
                    }
                }
                $lqipRel = "assets/dishes/{$hashBase}-lqip.jpg";
                $lqipAbs = public_path($lqipRel);
                if (!is_file($lqipAbs) || $force) {
                    try { $this->resizeToWidth($originalPath, $lqipAbs, 40, quality: 38); } catch(\Throwable $e) {}
                }
                $entry = [
                    'type' => 'dish',
                    'base' => $hashBase,
                    'variants' => $variantsMeta,
                    'lqip' => is_file($lqipAbs) ? $lqipRel : null,
                    'original' => 'storage/'.$rel,
                ];
                $manifest['dishes'][] = $entry;
                $count++;
            }
        });
        $this->info("Generated/validated variants for {$count} dishes");
    }

    protected function resizeToWidth(string $src, string $dest, int $targetWidth, int $quality = 74): void
    {
        $info = getimagesize($src);
        if (!$info) throw new \RuntimeException('Unrecognized image: '.$src);
        [$w,$h] = $info;
        if ($w <= 0 || $h <= 0) throw new \RuntimeException('Invalid dimensions: '.$src);
        $ratio = $targetWidth / $w;
        $tw = $targetWidth;
        $th = (int) round($h * $ratio);
        $image = imagecreatefromstring(file_get_contents($src));
        if (!$image) throw new \RuntimeException('Could not create image resource');
        $canvas = imagecreatetruecolor($tw,$th);
        imagecopyresampled($canvas, $image, 0,0,0,0,$tw,$th,$w,$h);
        if (!is_dir(dirname($dest))) mkdir(dirname($dest), 0775, true);
        imagejpeg($canvas, $dest, $quality);
        imagedestroy($image); imagedestroy($canvas);
    }

    protected function generateSliderVariants($disk, array &$manifest, bool $force): void
    {
        $sliderDir = trim(config('images.paths.sliders','sliders'),'/');
        $variantWidths = config('images.variants.sliders',[640,1024,1536,1920]);
        $absDir = $disk->path($sliderDir);
        if (!is_dir($absDir)) {
            // Attempt to create directory if missing
            if (!mkdir($absDir, 0775, true) && !is_dir($absDir)) {
                $this->warn('Unable to create sliders directory: '.$absDir);
                return;
            }
            $this->info('Created slider originals directory: '.$absDir.' (place original hero images here)');
        }
        $files = glob($absDir.'/*.{jpg,jpeg,png}', GLOB_BRACE) ?: [];
        $count = 0;
        foreach ($files as $file) {
            $base = pathinfo($file, PATHINFO_FILENAME);
            $variantsMeta = [];
            foreach ($variantWidths as $w) {
                $destRel = "assets/slider/{$base}-{$w}.jpg";
                $destAbs = public_path($destRel);
                if (is_file($destAbs) && !$force) {
                    [$width] = getimagesize($destAbs) ?: [null];
                    if ($width) $variantsMeta[] = ['width'=>$width,'jpg'=>$destRel];
                    continue;
                }
                try {
                    $this->resizeToWidth($file, $destAbs, $w, quality: 78);
                    [$width] = getimagesize($destAbs) ?: [$w];
                    $variantsMeta[] = ['width'=>$width,'jpg'=>$destRel];
                } catch (\Throwable $e) {
                    $this->warn('Slider variant gen failed: '.$e->getMessage());
                }
            }
            $lqipRel = "assets/slider/{$base}-lqip.jpg";
            $lqipAbs = public_path($lqipRel);
            if (!is_file($lqipAbs) || $force) {
                try { $this->resizeToWidth($file, $lqipAbs, 48, quality: 30); } catch(\Throwable $e) {}
            }
            $manifest['slider'][] = [
                'type' => 'slider',
                'base' => $base,
                'variants' => $variantsMeta,
                'lqip' => is_file($lqipAbs) ? $lqipRel : null,
                'original' => 'storage/'.$sliderDir.'/'.basename($file),
            ];
            $count++;
        }
        $this->info("Generated/validated slider variants for {$count} images");
    }
}
