<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\Dish;
use App\Models\Category;
use Spatie\Permission\Models\Role;

/**
 * One‑shot local environment bootstrap for manual testing / QA.
 * Drops & recreates schema (optional), seeds core data and prints test credentials.
 */
class LocalBootstrap extends Command
{
    protected $signature = 'local:bootstrap
        {--fresh : Drop all tables before migrating}
        {--no-seed : Skip running database seeders}
        {--force : Run even if APP_ENV=production (NOT recommended)}';

    protected $description = 'Rebuild the local database (optionally fresh) and seed test data with a credential summary';

    public function handle(): int
    {
        if (app()->environment('production') && !$this->option('force')) {
            $this->error('Refusing to run in production without --force');
            return self::FAILURE;
        }

        $fresh = (bool) $this->option('fresh');
        $seed  = !(bool) $this->option('no-seed');

        $this->line('<fg=blue>==> Local Bootstrap Start</>');
        $this->info('Connection: '.config('database.default'). ' DB: '.config('database.connections.'.config('database.default').'.database'));

        // Basic confirmation when dropping data
        if ($fresh) {
            if (!$this->confirm('This will DROP all tables. Continue?', true)) {
                $this->warn('Aborted.');
                return self::INVALID;
            }
            $this->call('migrate:fresh', ['--force' => true]);
        } else {
            // Ensure migrations run (safe if already up-to-date)
            $this->call('migrate', ['--force' => true]);
        }

        if ($seed) {
            $this->call('db:seed', ['--force' => true]);
        } else {
            $this->warn('Skipping seeders (per --no-seed)');
        }

        $this->newLine();
        $this->line('<fg=blue>==> Dataset Summary</>');

        $stats = [
            ['Metric','Count'],
            ['Users', (string) User::count()],
            ['Roles', (string) Role::count()],
            ['Categories', (string) Category::count()],
            ['Dishes', (string) Dish::count()],
        ];
        $this->table(['Metric','Count'], array_slice($stats, 1));

        // Collect test/demo users (seeders use @bonnydelight.ng domain)
        $testUsers = User::where('email', 'like', '%@bonnydelight.ng')->orderBy('id')->get();
        if ($testUsers->isNotEmpty()) {
            $this->line('<fg=blue>==> Test / Demo Accounts (local only)</>');
            $rows = [];
            foreach ($testUsers as $u) {
                $roles = $u->roles->pluck('name')->implode(',');
                $rows[] = [
                    $u->email,
                    $roles ?: $u->role,
                    $this->guessPasswordFor($u->email),
                ];
            }
            $this->table(['Email','Roles','Password (change in prod)'], $rows);
        } else {
            $this->warn('No @bonnydelight.ng test users found. Did you skip seeding?');
        }

        $this->line('<fg=blue>==> Feature Flags</>');
        try {
            $features = config('features.features');
            $rows = [];
            foreach ($features as $key => $cfg) {
                $rows[] = [
                    $key,
                    !empty($cfg['enabled']) ? 'ON' : 'off',
                    $cfg['roles'] ? implode(',', $cfg['roles']) : '-',
                ];
            }
            if ($rows) { $this->table(['Feature','Enabled','Scoped Roles'], $rows); }
        } catch (\Throwable $e) {
            $this->warn('Could not load features config: '.$e->getMessage());
        }

        $this->line('<fg=blue>==> Done</>');
        $this->info('You can now log in with any test user above.');
        return self::SUCCESS;
    }

    /**
     * Infer the seeded password from known patterns to avoid storing plain text.
     */
    protected function guessPasswordFor(string $email): string
    {
        // Map known seed emails to their plain password (from seeders) – local only convenience.
        return match($email) {
            'admin@bonnydelight.ng' => 'ChangeMe123!',
            'manager@bonnydelight.ng','kitchen@bonnydelight.ng','cashier@bonnydelight.ng',
            'editor@bonnydelight.ng','rider@bonnydelight.ng','customer@bonnydelight.ng' => 'TestPass123!',
            'admin.tester@bonnydelight.ng' => 'AdminTest123!',
            'user.tester@bonnydelight.ng' => 'UserTest123!',
            default => '***',
        };
    }
}
