<?php

namespace App\Console\Commands;

use App\Services\Inventory\ForecastingService;
use Illuminate\Console\Command;

class ProcessInventoryForecasting extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:process-forecasting 
                            {--skip-velocity : Skip velocity computation}
                            {--skip-alerts : Skip alert generation}
                            {--detailed : Show detailed output}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Complete inventory forecasting process - compute velocity and generate alerts';

    protected ForecastingService $forecastingService;

    /**
     * Create a new command instance.
     */
    public function __construct(ForecastingService $forecastingService)
    {
        parent::__construct();
        $this->forecastingService = $forecastingService;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('🎯 Starting complete inventory forecasting process...');
        $overallStartTime = microtime(true);
        
        try {
            $results = [
                'velocity' => null,
                'alerts' => null,
                'summary' => null,
            ];

            // Step 1: Compute Velocity
            if (!$this->option('skip-velocity')) {
                $results['velocity'] = $this->processVelocity();
            } else {
                $this->info('⏭️  Skipping velocity computation');
            }

            // Step 2: Generate Alerts
            if (!$this->option('skip-alerts')) {
                $results['alerts'] = $this->processAlerts();
            } else {
                $this->info('⏭️  Skipping alert generation');
            }

            // Step 3: Display Summary
            $results['summary'] = $this->displaySummary();

            $totalExecutionTime = round((microtime(true) - $overallStartTime) * 1000, 2);
            $this->info("🎉 Complete forecasting process finished in {$totalExecutionTime}ms");
            
            return Command::SUCCESS;
            
        } catch (\Exception $e) {
            $this->error("❌ Forecasting process failed: " . $e->getMessage());
            
            if ($this->option('detailed')) {
                $this->error($e->getTraceAsString());
            }
            
            return Command::FAILURE;
        }
    }

    /**
     * Process velocity computation
     */
    private function processVelocity(): array
    {
        $this->info('📊 Step 1: Computing velocity for all dishes...');
        $startTime = microtime(true);
        
        $results = $this->forecastingService->updateVelocityForAllDishes();
        
        $executionTime = round((microtime(true) - $startTime) * 1000, 2);
        
        if ($this->option('detailed')) {
            $this->table(
                ['Metric', 'Count'],
                [
                    ['Total processed', $results['processed']],
                    ['Successfully updated', $results['updated']],
                    ['Errors encountered', $results['errors']],
                ]
            );
        }

        if ($results['errors'] > 0) {
            $this->warn("⚠️  {$results['errors']} dishes failed velocity update");
        } else {
            $this->info("✅ Velocity computation completed ({$executionTime}ms)");
        }

        return $results;
    }

    /**
     * Process alert generation
     */
    private function processAlerts(): array
    {
        $this->info('🚨 Step 2: Generating inventory alerts...');
        $startTime = microtime(true);
        
        $results = $this->forecastingService->generateAlerts();
        
        $executionTime = round((microtime(true) - $startTime) * 1000, 2);
        
        if ($this->option('detailed')) {
            $this->table(
                ['Metric', 'Count'],
                [
                    ['Dishes processed', $results['processed']],
                    ['New alerts created', $results['alerts_created']],
                    ['Alerts resolved', $results['alerts_resolved']],
                ]
            );
        }

        if ($results['alerts_created'] > 0) {
            $this->warn("🆕 {$results['alerts_created']} new alerts created");
        }

        if ($results['alerts_resolved'] > 0) {
            $this->info("✅ {$results['alerts_resolved']} alerts auto-resolved");
        }

        $this->info("✅ Alert generation completed ({$executionTime}ms)");
        
        return $results;
    }

    /**
     * Display comprehensive summary
     */
    private function displaySummary(): array
    {
        $this->info('📋 Step 3: Generating forecasting summary...');
        
        // Get alert summary
        $alertSummary = $this->forecastingService->getCriticalAlertsSummary();
        
        // Get velocity statistics
        $velocityStats = $this->forecastingService->getVelocityStatistics();

        // Current inventory health
        $inventoryHealth = $this->calculateInventoryHealth($alertSummary, $velocityStats);

        $this->newLine();
        $this->info('🎯 INVENTORY FORECASTING SUMMARY');
        $this->info('═══════════════════════════════════');

        // Velocity Overview
        $this->table(
            ['Velocity Metrics', 'Value'],
            [
                ['Total dishes tracked', $velocityStats['total_dishes']],
                ['Dishes with sales activity', $velocityStats['dishes_with_sales']],
                ['Average 7-day velocity', "{$velocityStats['average_velocity_7d']} units/day"],
                ['Average 30-day velocity', "{$velocityStats['average_velocity_30d']} units/day"],
                ['Peak 7-day velocity', "{$velocityStats['max_velocity_7d']} units/day"],
            ]
        );

        // Alert Overview
        if ($alertSummary['total_active'] > 0) {
            $this->table(
                ['Alert Status', 'Count', 'Priority'],
                [
                    ['🔴 Critical Alerts', $alertSummary['critical'], 'IMMEDIATE ACTION'],
                    ['🟠 High Priority', $alertSummary['high'], 'Urgent'],
                    ['🟡 Medium Priority', $alertSummary['medium'], 'Monitor'],
                    ['🟢 Low Priority', $alertSummary['low'], 'Info'],
                    ['📊 Total Active', $alertSummary['total_active'], '-'],
                ]
            );

            // Alert breakdown by type
            $this->table(
                ['Alert Type', 'Count'],
                [
                    ['Out of Stock', $alertSummary['by_type']['out_of_stock']],
                    ['Low Stock', $alertSummary['by_type']['low_stock']],
                    ['Projected Stockout', $alertSummary['by_type']['projected_stockout']],
                    ['Safety Stock Breach', $alertSummary['by_type']['safety_stock_breach']],
                ]
            );
        } else {
            $this->info('✨ No active alerts - inventory is in excellent condition!');
        }

        // Inventory Health Score
        $this->info("🏥 Overall Inventory Health: {$inventoryHealth['score']}/100 ({$inventoryHealth['status']})");
        
        if (!empty($inventoryHealth['recommendations'])) {
            $this->warn('💡 Recommendations:');
            foreach ($inventoryHealth['recommendations'] as $recommendation) {
                $this->warn("   • {$recommendation}");
            }
        }

        return [
            'alert_summary' => $alertSummary,
            'velocity_stats' => $velocityStats,
            'health_score' => $inventoryHealth,
        ];
    }

    /**
     * Calculate overall inventory health score
     */
    private function calculateInventoryHealth(array $alertSummary, array $velocityStats): array
    {
        $score = 100;
        $recommendations = [];
        
        // Deduct points for alerts
        $score -= $alertSummary['critical'] * 20;  // Critical: -20 each
        $score -= $alertSummary['high'] * 10;      // High: -10 each
        $score -= $alertSummary['medium'] * 5;     // Medium: -5 each
        $score -= $alertSummary['low'] * 2;        // Low: -2 each
        
        // Deduct points for inactive dishes
        $inactiveRatio = $velocityStats['dishes_without_sales'] / max(1, $velocityStats['total_dishes']);
        $score -= $inactiveRatio * 15;  // Up to -15 for all inactive
        
        $score = max(0, round($score));
        
        // Determine status
        if ($score >= 90) {
            $status = 'EXCELLENT';
        } elseif ($score >= 75) {
            $status = 'GOOD';
        } elseif ($score >= 60) {
            $status = 'FAIR';
        } elseif ($score >= 40) {
            $status = 'POOR';
        } else {
            $status = 'CRITICAL';
        }
        
        // Generate recommendations
        if ($alertSummary['critical'] > 0) {
            $recommendations[] = "Address {$alertSummary['critical']} critical alerts immediately";
        }
        
        if (($alertSummary['by_type']['out_of_stock'] ?? 0) > 0) {
            $recommendations[] = "Restock {$alertSummary['by_type']['out_of_stock']} out-of-stock items";
        }
        
        if (($alertSummary['by_type']['projected_stockout'] ?? 0) > 0) {
            $recommendations[] = "Prepare restocking for {$alertSummary['by_type']['projected_stockout']} items predicted to run out";
        }
        
        if ($inactiveRatio > 0.3) {
            $recommendations[] = "Review {$velocityStats['dishes_without_sales']} dishes with no recent sales";
        }
        
        return [
            'score' => $score,
            'status' => $status,
            'recommendations' => $recommendations,
        ];
    }
}
