<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

/**
 * Rebuild dish image responsive JPEG variants and merge/update manifest.
 * This is a recovery tool if queued jobs failed or manifest was cleared.
 */
class RebuildDishImageVariants extends Command
{
    protected $signature = 'images:rebuild-dishes {--force : Overwrite existing generated variants}';
    protected $description = 'Rebuild responsive dish image variants and update the image manifest';

    protected array $widths = [200,400,600,800,1200,1600];

    public function handle(): int
    {
        $disk = Storage::disk('public');
        $files = collect($disk->files('dishes'))
            ->filter(fn($f)=>preg_match('/\.(jpe?g|png)$/i',$f))
            ->values();
        if ($files->isEmpty()) {
            $this->warn('No original dish images found in storage/dishes');
            return self::SUCCESS;
        }
        $outDir = public_path('assets/dishes');
        if (!is_dir($outDir)) mkdir($outDir, 0775, true);
        $manifestFile = public_path('assets/image-manifest.json');
        $manifest = [ 'generatedAt'=>now()->toIso8601String(), 'slider'=>[], 'dishes'=>[] ];
        if (is_file($manifestFile)) {
            try { $decoded = json_decode(file_get_contents($manifestFile), true); if (is_array($decoded)) { $manifest = array_merge(['slider'=>[], 'dishes'=>[]], $decoded); } } catch (\Throwable $e) {}
        }

        $count = 0; $force = $this->option('force');
        foreach ($files as $rel) {
            $path = $disk->path($rel);
            $base = pathinfo($path, PATHINFO_FILENAME);
            // Skip if looks like a generated variant (has -<digits>.jpg)
            if (preg_match('/-\d+\.(jpe?g|png)$/i', $rel)) continue;
            $imgInfo = @getimagesize($path);
            if (!$imgInfo) { $this->warn("Skip unreadable $rel"); continue; }
            [$wOrig,$hOrig] = [$imgInfo[0], $imgInfo[1]];
            $mime = $imgInfo['mime'] ?? '';
            $createFn = match($mime) {
                'image/jpeg' => 'imagecreatefromjpeg',
                'image/png' => 'imagecreatefrompng',
                default => null,
            };
            if (!$createFn || !is_callable($createFn)) { $this->warn("Unsupported format $rel"); continue; }
            $im = @$createFn($path); if (!$im) { $this->warn("Failed to read $rel"); continue; }
            $variants = [];
            foreach ($this->widths as $w) {
                if ($w >= $wOrig) continue;
                $h = (int) round($hOrig * ($w / $wOrig));
                $variantRel = "assets/dishes/{$base}-{$w}.jpg";
                $variantPath = public_path($variantRel);
                if (!$force && is_file($variantPath)) { $variants[] = ['width'=>$w,'jpg'=>$variantRel]; continue; }
                $canvas = imagecreatetruecolor($w,$h);
                imagecopyresampled($canvas,$im,0,0,0,0,$w,$h,$wOrig,$hOrig);
                imagejpeg($canvas,$variantPath,80);
                imagedestroy($canvas);
                $variants[] = ['width'=>$w,'jpg'=>$variantRel];
            }
            // LQIP
            $lqipRel = null; $lqipW = 24;
            if ($wOrig > 0) {
                $lqipH = (int) round($hOrig * ($lqipW / $wOrig));
                $lqipRel = "assets/dishes/{$base}-lqip.jpg";
                $lqipPath = public_path($lqipRel);
                if ($force || !is_file($lqipPath)) {
                    $tiny = imagecreatetruecolor($lqipW,$lqipH);
                    imagecopyresampled($tiny,$im,0,0,0,0,$lqipW,$lqipH,$wOrig,$hOrig);
                    imagejpeg($tiny,$lqipPath,35);
                    imagedestroy($tiny);
                }
            }
            imagedestroy($im);
            // Merge: remove older entry
            $manifest['dishes'] = array_values(array_filter($manifest['dishes'], fn($e)=>($e['base']??null)!==$base));
            $manifest['dishes'][] = [
                'type' => 'dish',
                'base' => $base,
                'variants' => $variants,
                'thumb' => null,
                'lqip' => $lqipRel,
                'original' => 'storage/dishes/'.basename($rel),
                'rebuilt' => true,
            ];
            $count++;
        }
        $manifest['generatedAt'] = now()->toIso8601String();
        file_put_contents($manifestFile, json_encode($manifest, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
        $this->info("Rebuilt variants for $count dish originals. Manifest updated.");
        return self::SUCCESS;
    }
}
