<?php

namespace App\Console\Commands;

use App\Services\Inventory\PerformanceService;
use Illuminate\Console\Command;

class ReconcileInventoryStats extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:reconcile-stats 
                            {--fix : Automatically fix discovered discrepancies}
                            {--report : Generate detailed discrepancy report}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Reconcile inventory stats with ledger to detect drift and inconsistencies';

    protected PerformanceService $performanceService;

    /**
     * Create a new command instance.
     */
    public function __construct(PerformanceService $performanceService)
    {
        parent::__construct();
        $this->performanceService = $performanceService;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('🔍 Starting inventory stats reconciliation...');
        $startTime = microtime(true);

        try {
            $results = $this->performanceService->reconcileStats();

            $this->displayResults($results);

            if ($this->option('report')) {
                $this->generateDetailedReport($results);
            }

            $executionTime = round((microtime(true) - $startTime) * 1000, 2);
            $this->info("✅ Reconciliation completed in {$executionTime}ms");
            
            return $this->determineExitCode($results);
            
        } catch (\Exception $e) {
            $this->error("❌ Reconciliation failed: " . $e->getMessage());
            return Command::FAILURE;
        }
    }

    /**
     * Display reconciliation results
     */
    private function displayResults(array $results): void
    {
        $this->table(
            ['Metric', 'Count'],
            [
                ['Dishes checked', $results['dishes_checked']],
                ['Discrepancies found', $results['discrepancies']],
                ['Automatically corrected', $results['corrected']],
                ['Errors encountered', count($results['errors'])],
            ]
        );

        // Status assessment
        if ($results['discrepancies'] === 0) {
            $this->info('✨ Perfect! No discrepancies found - all stats are in sync.');
        } elseif ($results['corrected'] === $results['discrepancies']) {
            $this->info("✅ All {$results['discrepancies']} discrepancies have been automatically corrected.");
        } else {
            $uncorrected = $results['discrepancies'] - $results['corrected'];
            $this->warn("⚠️  {$uncorrected} discrepancies require manual attention.");
        }

        // Show errors if any
        if (!empty($results['errors'])) {
            $this->error('🚨 Errors encountered during reconciliation:');
            foreach ($results['errors'] as $error) {
                $this->error("   • {$error}");
            }
        }
    }

    /**
     * Generate detailed discrepancy report
     */
    private function generateDetailedReport(array $results): void
    {
        $this->info('📋 Generating detailed reconciliation report...');
        
        // Get current discrepancies for detailed analysis
        $discrepancies = $this->findCurrentDiscrepancies();
        
        if (empty($discrepancies)) {
            $this->info('📊 No current discrepancies to report.');
            return;
        }

        $this->warn('📊 Detailed Discrepancy Report:');
        $this->table(
            ['Dish ID', 'Dish Name', 'Issue Type', 'Stats Value', 'Actual Value', 'Difference'],
            $discrepancies
        );

        // Generate recommendations
        $this->generateRecommendations($discrepancies);
    }

    /**
     * Find current discrepancies for reporting
     */
    private function findCurrentDiscrepancies(): array
    {
        $discrepancies = [];
        
        $stats = \App\Models\InventoryStats::with('dish:id,name,stock')->get();
        
        foreach ($stats as $stat) {
            $dish = $stat->dish;
            
            // Check stock discrepancy
            if ($stat->on_hand !== $dish->stock) {
                $discrepancies[] = [
                    $dish->id,
                    $dish->name ?? 'Unknown',
                    'Stock Level',
                    $stat->on_hand,
                    $dish->stock,
                    $dish->stock - $stat->on_hand,
                ];
            }

            // Check movement count
            $actualMovements = \App\Models\InventoryMovement::where('dish_id', $dish->id)->count();
            if ($stat->total_movements !== $actualMovements) {
                $discrepancies[] = [
                    $dish->id,
                    $dish->name ?? 'Unknown',
                    'Movement Count',
                    $stat->total_movements,
                    $actualMovements,
                    $actualMovements - $stat->total_movements,
                ];
            }

            // Check available calculation
            $expectedAvailable = max(0, $dish->stock - $stat->allocated);
            if ($stat->available !== $expectedAvailable) {
                $discrepancies[] = [
                    $dish->id,
                    $dish->name ?? 'Unknown',
                    'Available Stock',
                    $stat->available,
                    $expectedAvailable,
                    $expectedAvailable - $stat->available,
                ];
            }
        }

        return $discrepancies;
    }

    /**
     * Generate recommendations based on discrepancies
     */
    private function generateRecommendations(array $discrepancies): void
    {
        $recommendations = [];
        
        $stockIssues = collect($discrepancies)->where('2', 'Stock Level')->count();
        $movementIssues = collect($discrepancies)->where('2', 'Movement Count')->count();
        $availabilityIssues = collect($discrepancies)->where('2', 'Available Stock')->count();

        if ($stockIssues > 0) {
            $recommendations[] = "Run physical inventory count for {$stockIssues} dishes with stock discrepancies";
            $recommendations[] = "Review recent inventory movements for potential data entry errors";
        }

        if ($movementIssues > 0) {
            $recommendations[] = "Rebuild movement aggregates for {$movementIssues} dishes";
            $recommendations[] = "Check for orphaned or corrupted movement records";
        }

        if ($availabilityIssues > 0) {
            $recommendations[] = "Recalculate availability formulas for {$availabilityIssues} dishes";
            $recommendations[] = "Review allocation logic if implemented";
        }

        if (count($discrepancies) > 10) {
            $recommendations[] = "Consider full stats rebuild: inventory:update-stats --force";
        }

        if (count($discrepancies) > 20) {
            $recommendations[] = "Review data integrity and implement stronger validation rules";
        }

        if (!empty($recommendations)) {
            $this->warn('💡 Recommendations:');
            foreach ($recommendations as $recommendation) {
                $this->warn("   • {$recommendation}");
            }
        }
    }

    /**
     * Determine appropriate exit code based on results
     */
    private function determineExitCode(array $results): int
    {
        // Success if no discrepancies or all were corrected
        if ($results['discrepancies'] === 0 || $results['corrected'] === $results['discrepancies']) {
            return Command::SUCCESS;
        }

        // Warning if there are uncorrected discrepancies but no errors
        if (empty($results['errors'])) {
            return 1; // Warning exit code
        }

        // Failure if there were errors
        return Command::FAILURE;
    }
}
