<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Str;
use App\Models\Dish;

class RepairDishImages extends Command
{
    protected $signature = 'images:repair-dishes {--dry-run : Only report, do not write changes}';
    protected $description = 'Repair dish images by assigning existing assets for dishes with missing/unresolvable images';

    public function handle(): int
    {
        $dry = (bool) $this->option('dry-run');
        $root = public_path('assets/images');
        $countChecked = 0; $countFixed = 0; $countOk = 0; $countMissing = 0;

        $makeCandidates = function(string $name): array {
            $base = strtolower($name);
            $base = str_replace(['&','/','\\'], [' and ',' ',' '], $base);
            $base = preg_replace('/\s+/', ' ', $base);
            $dash = Str::slug($base, '-');
            $under = preg_replace('/[^a-z0-9]+/', '_', $base);
            $under = trim($under, '_');
            $compact = preg_replace('/[^a-z0-9]+/','', $base);
            $cands = ["{$dash}.jpg", "{$under}.jpg", "{$compact}.jpg"]; // common variants
            // Add some likely short forms
            $short = explode('-', $dash)[0] ?? $dash; if ($short) $cands[] = "$short.jpg";
            return array_values(array_unique($cands));
        };

        foreach (Dish::select('id','name','image')->orderBy('id')->cursor() as $dish) {
            $countChecked++;
            $url = $dish->public_image_url; // uses accessor
            if ($url) { $countOk++; continue; }

            $candidates = $makeCandidates($dish->name);
            $chosen = null;
            foreach ($candidates as $file) {
                $path = $root . DIRECTORY_SEPARATOR . $file;
                if (is_file($path)) { $chosen = 'assets/images/'.$file; break; }
            }

            if ($chosen) {
                if ($dry) {
                    $this->line("[DRY] Dish #{$dish->id} '{$dish->name}' -> set image='{$chosen}'");
                } else {
                    $dish->image = $chosen;
                    $dish->save();
                    $this->info("Fixed Dish #{$dish->id} '{$dish->name}' -> image='{$chosen}'");
                    $countFixed++;
                }
            } else {
                $this->warn("Missing image for Dish #{$dish->id} '{$dish->name}' (no asset match)");
                $countMissing++;
            }
        }

        $this->line("");
        $this->table(['Checked','OK','Fixed','Still Missing'], [[
            $countChecked, $countOk, $countFixed, $countMissing
        ]]);

        return Command::SUCCESS;
    }
}
