<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\Reporting\ReportingService;
use Carbon\Carbon;

class ReportingRebuildCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'reporting:rebuild 
                            {window=daily : Reporting window (daily, weekly, monthly)}
                            {--clear-cache : Clear existing cache before rebuilding}
                            {--start-date= : Start date for custom range (YYYY-MM-DD)}
                            {--end-date= : End date for custom range (YYYY-MM-DD)}
                            {--all : Rebuild all windows}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Rebuild reporting metrics and KPIs. Idempotent - safe to run multiple times.';

    /**
     * Execute the console command.
     */
    public function handle(ReportingService $reportingService): int
    {
        $this->info('🔄 Starting reporting rebuild...');
        
        if ($this->option('clear-cache')) {
            $this->info('🗑️ Clearing existing cache...');
            $reportingService->clearCache();
        }

        $windows = $this->option('all') ? ['daily', 'weekly', 'monthly'] : [$this->argument('window')];
        
        $startDate = $this->option('start-date') ? Carbon::parse($this->option('start-date')) : null;
        $endDate = $this->option('end-date') ? Carbon::parse($this->option('end-date')) : null;

        if ($startDate && $endDate) {
            $this->info("📅 Using custom date range: {$startDate->toDateString()} to {$endDate->toDateString()}");
        }

        foreach ($windows as $window) {
            $this->info("📊 Processing {$window} window...");
            
            try {
                $report = $reportingService->generateKPIReport($window, $startDate, $endDate);
                $topSellers = $reportingService->getTopSellingDishes($window, 5, $startDate, $endDate);
                
                $this->displayReport($window, $report, $topSellers);
                
            } catch (\Exception $e) {
                $this->error("❌ Failed to generate {$window} report: " . $e->getMessage());
                return Command::FAILURE;
            }
        }

        $this->info('✅ Reporting rebuild completed successfully!');
        return Command::SUCCESS;
    }

    /**
     * Display report summary in console
     */
    private function displayReport(string $window, array $report, array $topSellers): void
    {
        $this->newLine();
        $this->line("🏷️  <fg=cyan>{$window} Report ({$report['period']['start_date']} to {$report['period']['end_date']})</>");
        
        // Sales metrics
        $sales = $report['sales_metrics'];
        $this->line("💰 <fg=green>Sales:</> Gross: ₦{$sales['gross_sales']}, Net: ₦{$sales['net_sales']}, AOV: ₦{$sales['average_order_value']}");
        $this->line("📈 <fg=blue>Orders:</> {$sales['total_orders']} orders, {$sales['total_items_sold']} items, {$sales['sales_velocity_per_day']} items/day");
        
        // Inventory metrics
        $inventory = $report['inventory_metrics'];
        $grossMargin = $inventory['cogs'] > 0 ? (($sales['net_sales'] - $inventory['cogs']) / $sales['net_sales']) * 100 : 0;
        $this->line("📦 <fg=yellow>Inventory:</> ₦{$inventory['current_inventory_value']} value, {$inventory['total_on_hand']} units, {$inventory['days_of_inventory']} days runway");
        $this->line("📊 <fg=magenta>Performance:</> {$grossMargin}% gross margin, {$inventory['inventory_turnover']} turnover");
        
        // Operational metrics
        $ops = $report['operational_metrics'];
        $this->line("⚡ <fg=red>Operations:</> {$ops['fill_rate_percent']}% fill rate, {$ops['cancellation_rate_percent']}% cancellation rate");
        
        // Top sellers
        if (!empty($topSellers)) {
            $this->line("🏆 <fg=green>Top Sellers:</>");
            foreach (array_slice($topSellers, 0, 3) as $i => $seller) {
                $this->line("   " . ($i + 1) . ". {$seller['dish_name']}: {$seller['total_sold']} sold (₦{$seller['revenue']})");
            }
        }
        
        $this->newLine();
    }
}
