<?php

namespace App\Console\Commands;

use App\Services\Inventory\PerformanceService;
use Illuminate\Console\Command;

class UpdateInventoryStats extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:update-stats 
                            {--dish-id= : Update stats for specific dish ID}
                            {--force : Force update even if recently updated}
                            {--benchmark : Run performance benchmarks after update}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update cached inventory statistics for performance optimization';

    protected PerformanceService $performanceService;

    /**
     * Create a new command instance.
     */
    public function __construct(PerformanceService $performanceService)
    {
        parent::__construct();
        $this->performanceService = $performanceService;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('📊 Starting inventory stats update...');
        $startTime = microtime(true);

        try {
            $dishId = $this->option('dish-id');
            
            if ($dishId) {
                $this->updateSpecificDish($dishId);
            } else {
                $this->updateAllDishes();
            }

            // Run benchmarks if requested
            if ($this->option('benchmark')) {
                $this->runBenchmarks();
            }

            $executionTime = round((microtime(true) - $startTime) * 1000, 2);
            $this->info("✅ Stats update completed in {$executionTime}ms");
            
            return Command::SUCCESS;
            
        } catch (\Exception $e) {
            $this->error("❌ Stats update failed: " . $e->getMessage());
            return Command::FAILURE;
        }
    }

    /**
     * Update stats for a specific dish
     */
    private function updateSpecificDish(int $dishId): void
    {
        $dish = \App\Models\Dish::find($dishId);
        
        if (!$dish) {
            $this->error("❌ Dish with ID {$dishId} not found");
            return;
        }

        $this->info("📈 Updating stats for dish: {$dish->name} (ID: {$dishId})");
        
        $stats = $this->performanceService->updateDishStats($dish);
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['On Hand', $stats->on_hand],
                ['Available', $stats->available],
                ['Total Sales', $stats->total_sales],
                ['Total Movements', $stats->total_movements],
                ['7-day Velocity', "{$stats->velocity_7d} units/day"],
                ['Sales Value', "₦" . number_format($stats->total_sales_value, 2)],
                ['Inventory Value', "₦" . number_format($stats->inventory_value, 2)],
                ['Last Updated', $stats->stats_calculated_at?->format('Y-m-d H:i:s')],
            ]
        );
    }

    /**
     * Update stats for all dishes
     */
    private function updateAllDishes(): void
    {
        $this->info('📈 Updating stats for all dishes...');
        
        $results = $this->performanceService->updateAllInventoryStats();
        
        $this->table(
            ['Metric', 'Count'],
            [
                ['Total processed', $results['processed']],
                ['Records updated', $results['updated']],
                ['Records created', $results['created']],
                ['Errors encountered', $results['errors']],
            ]
        );

        if ($results['errors'] > 0) {
            $this->warn("⚠️  {$results['errors']} dishes failed to update. Check logs for details.");
        }

        // Show summary statistics
        $this->info('📊 Stats Summary:');
        $this->displayStatsSummary();
    }

    /**
     * Display summary of current stats
     */
    private function displayStatsSummary(): void
    {
        $stats = \App\Models\InventoryStats::selectRaw('
            COUNT(*) as total_dishes,
            SUM(on_hand) as total_stock,
            SUM(available) as total_available,
            SUM(total_sales) as total_sales,
            SUM(total_sales_value) as total_sales_value,
            SUM(inventory_value) as total_inventory_value,
            AVG(velocity_7d) as avg_velocity_7d,
            COUNT(CASE WHEN on_hand > 0 THEN 1 END) as dishes_with_stock,
            COUNT(CASE WHEN on_hand <= 0 THEN 1 END) as dishes_out_of_stock
        ')->first();

        $this->table(
            ['Metric', 'Value'],
            [
                ['Total dishes tracked', $stats->total_dishes],
                ['Dishes with stock', $stats->dishes_with_stock],
                ['Dishes out of stock', $stats->dishes_out_of_stock],
                ['Total stock units', number_format($stats->total_stock)],
                ['Total available units', number_format($stats->total_available)],
                ['Total sales (lifetime)', number_format($stats->total_sales)],
                ['Total sales value', "₦" . number_format($stats->total_sales_value, 2)],
                ['Total inventory value', "₦" . number_format($stats->total_inventory_value, 2)],
                ['Average 7-day velocity', round($stats->avg_velocity_7d, 2) . " units/day"],
            ]
        );
    }

    /**
     * Run performance benchmarks
     */
    private function runBenchmarks(): void
    {
        $this->info('🏃 Running performance benchmarks...');
        
        $benchmarks = $this->performanceService->getPerformanceBenchmarks();
        
        $this->table(
            ['Query', 'Time (ms)', 'Description'],
            [
                ['Available Dishes', $benchmarks['available_dishes_query']['time_ms'], $benchmarks['available_dishes_query']['description']],
                ['Low Stock Items', $benchmarks['low_stock_query']['time_ms'], $benchmarks['low_stock_query']['description']],
                ['Velocity Statistics', $benchmarks['velocity_stats_query']['time_ms'], $benchmarks['velocity_stats_query']['description']],
                ['Recent Snapshots', $benchmarks['recent_snapshots_query']['time_ms'], $benchmarks['recent_snapshots_query']['description']],
            ]
        );

        $this->info("📊 Total benchmark time: {$benchmarks['total_benchmark_time']}ms");
        
        // Performance assessment
        $avgTime = collect($benchmarks)->except('total_benchmark_time')->avg(fn($b) => $b['time_ms']);
        if ($avgTime < 10) {
            $this->info("✅ Performance: EXCELLENT (avg: " . round($avgTime, 2) . "ms)");
        } elseif ($avgTime < 50) {
            $this->info("🟢 Performance: GOOD (avg: " . round($avgTime, 2) . "ms)");
        } elseif ($avgTime < 100) {
            $this->warn("🟡 Performance: FAIR (avg: " . round($avgTime, 2) . "ms)");
        } else {
            $this->error("🔴 Performance: POOR (avg: " . round($avgTime, 2) . "ms) - Consider optimization");
        }
    }
}
