<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        // Run complete inventory forecasting process every night at 2 AM
        $schedule->command('inventory:process-forecasting')
                 ->dailyAt('02:00')
                 ->name('inventory-forecasting')
                 ->description('Complete inventory forecasting: velocity computation and alert generation')
                 ->withoutOverlapping()
                 ->runInBackground();

        // Run additional alert check during business hours (every 4 hours from 8 AM to 8 PM)
        $schedule->command('inventory:generate-alerts')
                 ->cron('0 8,12,16,20 * * *')
                 ->name('inventory-alerts-business-hours')
                 ->description('Generate inventory alerts during business hours')
                 ->withoutOverlapping();

        // Weekly velocity recalculation (more intensive)
        $schedule->command('inventory:compute-velocity')
                 ->weeklyOn(1, '01:00') // Monday at 1 AM
                 ->name('weekly-velocity-update')
                 ->description('Weekly comprehensive velocity recalculation')
                 ->withoutOverlapping()
                 ->runInBackground();

        // ===== PHASE 8: PERFORMANCE & SNAPSHOT LAYER =====
        
        // Update inventory stats every 4 hours
        $schedule->command('inventory:update-stats')
                 ->cron('0 */4 * * *')
                 ->name('inventory-stats-update')
                 ->description('Update cached inventory statistics for performance')
                 ->withoutOverlapping()
                 ->runInBackground();

        // Create daily snapshots at 3 AM (after forecasting)
        $schedule->command('inventory:create-snapshots')
                 ->dailyAt('03:00')
                 ->name('daily-inventory-snapshots')
                 ->description('Create daily inventory snapshots for historical analysis')
                 ->withoutOverlapping()
                 ->runInBackground();

        // Reconcile stats weekly to detect drift
        $schedule->command('inventory:reconcile-stats --fix')
                 ->weeklyOn(1, '04:00') // Monday at 4 AM
                 ->name('weekly-stats-reconciliation')
                 ->description('Weekly reconciliation to detect and fix stats drift')
                 ->withoutOverlapping()
                 ->runInBackground();

        // Performance benchmark monthly
        $schedule->command('inventory:update-stats --benchmark')
                 ->monthlyOn(1, '05:00') // 1st of month at 5 AM
                 ->name('monthly-performance-benchmark')
                 ->description('Monthly performance benchmarking and optimization check')
                 ->withoutOverlapping();
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
    $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}
