<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Dish;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class AdminAnalyticsController extends Controller
{
    public function index(Request $request)
    {
        return $this->__invoke($request);
    }

    public function __invoke(Request $request)
    {
        // Ensure only admin roles can access
        abort_unless(auth()->user()->hasAnyRole(['Super Admin', 'Manager']), 403);
        
        // Business analytics - admin only
        $today = now()->startOfDay();
        $thisMonth = now()->startOfMonth();
        
        $analytics = [
            'orders_today' => Order::whereDate('created_at', $today)->count(),
            'revenue_today' => Order::whereDate('created_at', $today)
                ->whereNotIn('status', ['Cancelled'])
                ->sum('total'),
            'orders_month' => Order::whereDate('created_at', '>=', $thisMonth)->count(),
            'revenue_month' => Order::whereDate('created_at', '>=', $thisMonth)
                ->whereNotIn('status', ['Cancelled'])
                ->sum('total'),
            'AOV_month' => Order::whereDate('created_at', '>=', $thisMonth)
                ->whereNotIn('status', ['Cancelled'])
                ->avg('total'),
        ];
        
        // Top dishes by sales volume - admin view
        $topDishes = Dish::select(
                'dishes.id',
                'dishes.category_id',
                'dishes.name',
                'dishes.description',
                'dishes.price',
                'dishes.image',
                'dishes.vegetarian',
                'dishes.halal',
                'dishes.gluten_free',
                'dishes.created_at',
                'dishes.updated_at',
                DB::raw('COUNT(order_items.id) as order_count')
            )
            ->join('order_items', 'dishes.id', '=', 'order_items.dish_id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->whereDate('orders.created_at', '>=', $thisMonth)
            ->whereNotIn('orders.status', ['Cancelled'])
            ->groupBy(
                'dishes.id',
                'dishes.category_id',
                'dishes.name',
                'dishes.description',
                'dishes.price',
                'dishes.image',
                'dishes.vegetarian',
                'dishes.halal',
                'dishes.gluten_free',
                'dishes.created_at',
                'dishes.updated_at'
            )
            ->orderBy('order_count', 'desc')
            ->limit(5)
            ->get();
        
        // Recent orders for admin overview
        $recentOrders = Order::with(['user', 'items.dish'])
            ->latest()
            ->limit(10)
            ->get();
        
        // Customer analytics (users without admin roles)
        $customerStats = [
            'total_customers' => User::whereDoesntHave('roles')->count(),
            'new_customers_month' => User::whereDoesntHave('roles')
                ->whereDate('created_at', '>=', $thisMonth)
                ->count(),
        ];

        // Image diagnostics: find dishes whose public image URL cannot be resolved
        $missingImages = Dish::select('id','name','image')
            ->get()
            ->filter(function($d){ return empty($d->public_image_url); })
            ->values();
        
        return view('admin.analytics', compact(
            'analytics',
            'topDishes',
            'recentOrders',
            'customerStats',
            'missingImages'
        ));
    }
}