<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Dish;
use App\Models\InventoryMovement;
use App\Services\Inventory\InventoryService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Validation\ValidationException;
use Carbon\Carbon;

class InventoryController extends Controller
{
    public function index()
    {
        // Calculate KPIs for inventory dashboard
        $totalDishes = Dish::count();
        $lowStockCount = Dish::whereRaw('stock <= low_stock_threshold')->count();
        $totalOnHand = Dish::sum('stock');
        
        // Top selling dishes (last 30 days based on sales movements) - optimized query
        $topSellers = DB::table('inventory_movements')
            ->join('dishes', 'inventory_movements.dish_id', '=', 'dishes.id')
        ->select('dishes.id', 'dishes.name', 'dishes.stock', 
            DB::raw('SUM(ABS(`inventory_movements`.`change`)) as total_sold'))
            ->where('inventory_movements.reason', 'sale')
            ->where('inventory_movements.created_at', '>=', Carbon::now()->subDays(30))
            ->whereNull('dishes.deleted_at') // Exclude soft deleted dishes
            ->groupBy('dishes.id', 'dishes.name', 'dishes.stock')
            ->orderByDesc('total_sold')
            ->limit(10)
            ->get();
        
        // Recent movements (last 10) - select only needed columns
        $recentMovements = InventoryMovement::select('id', 'dish_id', 'change', 'reason', 'user_id', 'created_at')
            ->with(['dish:id,name', 'user:id,name'])
            ->latest('created_at')
            ->limit(10)
            ->get();
        
        // Low stock dishes (using individual thresholds) - optimized query
        $lowStockDishes = Dish::select('id', 'name', 'stock', 'low_stock_threshold')
            ->whereRaw('stock <= low_stock_threshold')
            ->orderBy('stock', 'asc')
            ->limit(10)
            ->get();
        
        // Stock value (basic calculation)
        $totalStockValue = Dish::whereRaw('stock > 0')
            ->selectRaw('SUM(stock * price) as total_value')
            ->value('total_value') ?? 0;
        
        return view('admin.inventory.index', compact(
            'totalDishes',
            'lowStockCount', 
            'totalOnHand',
            'topSellers',
            'recentMovements',
            'lowStockDishes',
            'totalStockValue'
        ));
    }

    public function movements(Request $request)
    {
        // Validate pagination size
        $perPage = min((int)$request->get('per_page', 20), 100); // Max 100 items per page
        
    $query = InventoryMovement::with(['dish:id,name', 'user:id,name', 'order:id', 'location:id,name']);
        
        // Apply filters efficiently using indexes
        if ($request->filled('dish_id')) {
            $query->where('dish_id', $request->dish_id);
        }
        
        if ($request->filled('reason')) {
            $query->where('reason', $request->reason);
        }
        
        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }
        if ($request->filled('location_id')) {
            $query->where('location_id', $request->location_id);
        }
        
        if ($request->filled('order_id')) {
            $query->where('order_id', $request->order_id);
        }
        
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->whereHas('dish', function($dishQuery) use ($search) {
                    $dishQuery->where('name', 'like', "%{$search}%");
                })
                ->orWhere('note', 'like', "%{$search}%")
                ->orWhere('reason', 'like', "%{$search}%");
            });
        }
        
        $movements = $query->latest('created_at')->paginate($perPage)->withQueryString();
        
        // Get filter options for dropdowns - only load what's needed
        $dishes = Dish::select('id', 'name')->orderBy('name')->get();
    $users = \App\Models\User::select('id', 'name')->orderBy('name')->get();
    $locations = \App\Models\Location::select('id','name')->orderBy('name')->get();
        $inventoryService = app(InventoryService::class);
        $reasons = $inventoryService->getValidReasons();
        
        return view('admin.inventory.movements', compact(
            'movements', 
            'dishes', 
            'users', 
            'reasons',
            'locations',
            'request'
        ));
    }

    public function exportMovements(Request $request)
    {
    $query = InventoryMovement::with(['dish', 'user', 'order', 'location']);
        
        // Apply the same filters as the movements view
        if ($request->filled('dish_id')) {
            $query->where('dish_id', $request->dish_id);
        }
        
        if ($request->filled('reason')) {
            $query->where('reason', $request->reason);
        }
        
        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }
        if ($request->filled('location_id')) {
            $query->where('location_id', $request->location_id);
        }
        
        if ($request->filled('order_id')) {
            $query->where('order_id', $request->order_id);
        }
        
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->whereHas('dish', function($dishQuery) use ($search) {
                    $dishQuery->where('name', 'like', "%{$search}%");
                })
                ->orWhere('note', 'like', "%{$search}%")
                ->orWhere('reason', 'like', "%{$search}%");
            });
        }
        
        $movements = $query->latest()->get();
        
        // Generate CSV filename with timestamp and filters
        $filename = 'inventory_movements_' . now()->format('Y-m-d_His');
        if ($request->filled('date_from') || $request->filled('date_to')) {
            $filename .= '_' . ($request->date_from ?? 'start') . '_to_' . ($request->date_to ?? 'end');
        }
        $filename .= '.csv';
        
        // Set headers for CSV download
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        return response()->stream(function() use ($movements) {
            $handle = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($handle, [
                'ID',
                'Date',
                'Time', 
                'Dish ID',
                'Dish Name',
                'Change',
                'Reason',
                'Location ID',
                'Location Name',
                'User ID',
                'User Name',
                'Order ID',
                'Note',
                'Created At'
            ]);
            
            // Add data rows
            foreach ($movements as $movement) {
                fputcsv($handle, [
                    $movement->id,
                    $movement->created_at->format('Y-m-d'),
                    $movement->created_at->format('H:i:s'),
                    $movement->dish_id,
                    $movement->dish->name ?? 'Unknown Dish',
                    $movement->change,
                    $movement->reason,
                    $movement->location_id,
                    optional($movement->location)->name,
                    $movement->user_id,
                    $movement->user->name ?? 'System',
                    $movement->order_id,
                    $movement->note,
                    $movement->created_at->toISOString()
                ]);
            }
            
            fclose($handle);
        }, 200, $headers);
    }

    // Per-location availability aggregation view
    public function availability(Request $request)
    {
        // Filters
        $locationId = $request->get('location_id');
        $search = trim((string)$request->get('search'));
        $perPage = min((int)$request->get('per_page', 25), 100);

        // Build aggregate: sum of movements per dish and location if using ledger; fallback to dish.stock for total
        $query = DB::table('inventory_movements as im')
            ->join('dishes as d', 'd.id', '=', 'im.dish_id')
            ->leftJoin('locations as l', 'l.id', '=', 'im.location_id')
            ->select(
                'd.id as dish_id', 'd.name as dish_name', 'd.price', 'd.low_stock_threshold',
                'l.id as location_id', 'l.name as location_name',
                DB::raw('COALESCE(SUM(`im`.`change`), 0) as qty')
            )
            ->groupBy('d.id', 'd.name', 'd.price', 'd.low_stock_threshold', 'l.id', 'l.name')
            ->orderBy('d.name')
            ->orderBy('l.name');

        if ($locationId) {
            $query->where('im.location_id', $locationId);
        }
        if ($search !== '') {
            $query->where(function($q) use ($search) {
                $q->where('d.name', 'like', "%$search%")
                  ->orWhere('l.name', 'like', "%$search%");
            });
        }

        // Clone filter logic for grand totals across full dataset
        $grandTotalsQuery = DB::table('inventory_movements as im')
            ->join('dishes as d', 'd.id', '=', 'im.dish_id')
            ->leftJoin('locations as l', 'l.id', '=', 'im.location_id');
        if ($locationId) { $grandTotalsQuery->where('im.location_id', $locationId); }
        if ($search !== '') {
            $grandTotalsQuery->where(function($q) use ($search) {
                $q->where('d.name', 'like', "%$search%")
                  ->orWhere('l.name', 'like', "%$search%");
            });
        }
        $grandTotals = $grandTotalsQuery->select(
                DB::raw('COALESCE(SUM(`im`.`change`),0) as qty'),
                DB::raw('COALESCE(SUM((CASE WHEN `im`.`change`>0 THEN `im`.`change` ELSE 0 END) * d.price),0) as pos_value'),
                DB::raw('COALESCE(SUM(ABS(CASE WHEN `im`.`change`<0 THEN `im`.`change` ELSE 0 END) * d.price),0) as neg_value')
            )->first();

        // Current valuation snapshot using dish.stock (optionally filter by dish name)
        $stockValQuery = DB::table('dishes');
        if ($search !== '') {
            $stockValQuery->where('name', 'like', "%$search%");
        }
        $currentValuation = (float) ($stockValQuery->selectRaw('COALESCE(SUM(COALESCE(stock,0) * price),0) as val')->value('val') ?? 0);

        // Paginate manually via LengthAwarePaginator because of groupBy
        $page = (int)max(1, $request->get('page', 1));
        $all = $query->get();
        $total = $all->count();
        $items = $all->forPage($page, $perPage)->values();
        $paginator = new \Illuminate\Pagination\LengthAwarePaginator($items, $total, $perPage, $page, [
            'path' => url()->current(),
            'query' => $request->query(),
        ]);

        // Datasets for filters and quick nav
        $locations = \App\Models\Location::orderBy('name')->get(['id','name']);

        return view('admin.inventory.availability', [
            'rows' => $paginator,
            'locations' => $locations,
            'request' => $request,
            'grandTotals' => $grandTotals,
            'currentValuation' => $currentValuation,
        ]);
    }

    // CSV export for availability aggregates
    public function exportAvailability(Request $request)
    {
        $locationId = $request->get('location_id');
        $search = trim((string)$request->get('search'));

        $query = DB::table('inventory_movements as im')
            ->join('dishes as d', 'd.id', '=', 'im.dish_id')
            ->leftJoin('locations as l', 'l.id', '=', 'im.location_id')
            ->select(
                'd.id as dish_id', 'd.name as dish_name', 'd.price', 'd.low_stock_threshold',
                'l.id as location_id', 'l.name as location_name',
                DB::raw('COALESCE(SUM(`im`.`change`), 0) as qty')
            )
            ->groupBy('d.id', 'd.name', 'd.price', 'd.low_stock_threshold', 'l.id', 'l.name')
            ->orderBy('d.name')
            ->orderBy('l.name');

        if ($locationId) {
            $query->where('im.location_id', $locationId);
        }
        if ($search !== '') {
            $query->where(function($q) use ($search) {
                $q->where('d.name', 'like', "%$search%")
                  ->orWhere('l.name', 'like', "%$search%");
            });
        }

        $rows = $query->get();

        $filename = 'inventory_availability_' . now()->format('Y-m-d_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        return response()->stream(function() use ($rows) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['Dish ID','Dish Name','Location ID','Location Name','Qty','Threshold','Price','Value']);
            foreach ($rows as $r) {
                $qty = (int) $r->qty;
                $price = (float) $r->price;
                fputcsv($handle, [
                    $r->dish_id,
                    $r->dish_name,
                    $r->location_id,
                    $r->location_name,
                    $qty,
                    (int)$r->low_stock_threshold,
                    $price,
                    $qty * $price,
                ]);
            }
            fclose($handle);
        }, 200, $headers);
    }

    // JSON endpoint: All-locations totals per requested dish IDs
    public function availabilityTotals(Request $request)
    {
        $ids = (array) $request->get('dish_ids', []);
        $ids = array_values(array_filter(array_map('intval', $ids)));
        if (empty($ids)) {
            return response()->json(['totals' => []]);
        }

        $cacheKey = 'availability:totals:' . implode(',', $ids);
        $rows = cache()->remember($cacheKey, now()->addSeconds(10), function() use ($ids) {
            return DB::table('inventory_movements as im')
                ->select('im.dish_id', DB::raw('COALESCE(SUM(`im`.`change`), 0) as qty'))
                ->whereIn('im.dish_id', $ids)
                ->groupBy('im.dish_id')
                ->get();
        });

        return response()->json(['totals' => $rows]);
    }

    // JSON endpoint: sparkline series of last N movement deltas for a dish
    public function availabilitySparkline(Request $request)
    {
        $dishId = (int) $request->get('dish_id');
        $n = min(max((int)$request->get('n', 20), 5), 100);
        if ($dishId <= 0) return response()->json(['series' => []]);

        $cacheKey = "availability:sparkline:{$dishId}:{$n}";
        $movs = cache()->remember($cacheKey, now()->addSeconds(10), function() use ($dishId, $n) {
            return DB::table('inventory_movements')
                ->where('dish_id', $dishId)
                ->orderByDesc('id')
                ->limit($n)
                ->pluck('change')
                ->toArray();
        });

        // Return chronological order for drawing
        $series = array_reverse($movs);
        return response()->json(['series' => $series]);
    }

    // Pivot-style grid (dishes x locations) for small number of locations
    public function availabilityPivot(Request $request)
    {
        $search = trim((string)$request->get('search'));
        $perPage = min((int)$request->get('per_page', 50), 200);

        // small set of locations (e.g., first 10 active)
        $locations = \App\Models\Location::where('is_active', true)
            ->orderBy('name')
            ->limit(10)
            ->get(['id','name']);

        // select dishes filtered by search
        $dishQuery = DB::table('dishes')->select('id','name','price','low_stock_threshold');
        if ($search !== '') {
            $dishQuery->where('name', 'like', "%$search%");
        }
        $dishQuery->orderBy('name');
        $page = (int)max(1, $request->get('page', 1));
        $allDishes = $dishQuery->get();
        $total = $allDishes->count();
        $dishes = $allDishes->forPage($page, $perPage)->values();
        $paginator = new \Illuminate\Pagination\LengthAwarePaginator($dishes, $total, $perPage, $page, [
            'path' => url()->current(),
            'query' => $request->query(),
        ]);

        // movement sums per dish x location
        $dishIds = $dishes->pluck('id')->all();
        $matrix = [];
        if (!empty($dishIds) && $locations->count()) {
            $movs = DB::table('inventory_movements')
                ->select('dish_id','location_id', DB::raw('COALESCE(SUM(`change`),0) as qty'))
                ->whereIn('dish_id', $dishIds)
                ->whereIn('location_id', $locations->pluck('id')->all())
                ->groupBy('dish_id','location_id')
                ->get();
            foreach ($movs as $m) {
                $matrix[$m->dish_id][$m->location_id] = (int)$m->qty;
            }
        }

        return view('admin.inventory.availability-pivot', [
            'dishes' => $paginator,
            'locations' => $locations,
            'matrix' => $matrix,
            'request' => $request,
        ]);
    }

    public function history(Dish $dish)
    {
        if (!Schema::hasColumn('dishes', 'stock')) {
            return redirect()->route('admin.dishes.index')->with('status', 'Inventory not initialized yet. Please run migrations.');
        }
        
        $movements = InventoryMovement::where('dish_id', $dish->id)
            ->with(['user', 'order'])
            ->latest()
            ->paginate(20);
        
        return view('admin.inventory.history', compact('dish', 'movements'));
    }

    public function adjust(Request $request, Dish $dish)
    {
        if (!Schema::hasColumn('dishes', 'stock')) {
            return back()->withErrors(['stock' => 'Inventory not initialized yet. Please run database migrations.']);
        }
        
        $data = $request->validate([
            'change' => 'required|integer',
            'reason' => 'nullable|string|max:100',
            'note' => 'nullable|string|max:1000',
        ]);

        $inventoryService = app(InventoryService::class);
        $change = (int) $data['change'];
        $reason = $data['reason'] ?? 'adjustment';
        
        // Validate reason against canonical list
        if (!$inventoryService->isValidReason($reason)) {
            return back()->withErrors(['reason' => 'Invalid reason. Valid reasons: ' . implode(', ', array_keys($inventoryService->getValidReasons()))])->withInput();
        }

        try {
            $inventoryService->adjust(
                dish: $dish,
                change: $change,
                reason: $reason,
                user: Auth::user(),
                note: $data['note'] ?? null
            );
            
            return back()->with('status', 'Stock updated successfully');
        } catch (ValidationException $e) {
            return back()->withErrors($e->errors())->withInput();
        }
    }

    // Show transfer form
    public function transferForm(Request $request)
    {
        $dishes = Dish::orderBy('name')->get(['id','name','stock']);
        $locations = \App\Models\Location::orderBy('name')->get(['id','name','code']);
        return view('admin.inventory.transfer', compact('dishes','locations'));
    }

    // Handle transfer submission
    public function transferStore(Request $request)
    {
        $data = $request->validate([
            'dish_id' => 'required|exists:dishes,id',
            'from_location_id' => 'required|exists:locations,id|different:to_location_id',
            'to_location_id' => 'required|exists:locations,id',
            'quantity' => 'required|integer|min:1',
            'note' => 'nullable|string|max:500',
        ]);

        $dish = Dish::findOrFail($data['dish_id']);
        $from = \App\Models\Location::findOrFail($data['from_location_id']);
        $to = \App\Models\Location::findOrFail($data['to_location_id']);
        $qty = (int)$data['quantity'];

        $svc = app(InventoryService::class);
        try {
            $svc->transfer($dish, $qty, $from, $to, Auth::user(), $data['note'] ?? null);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return back()->withErrors(['transfer' => $e->getMessage()])->withInput();
        }

        return redirect()->route('admin.inventory.movements')->with('status', 'Transfer recorded');
    }
}
