<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\Reporting\ReportingService;
use Carbon\Carbon;

class ReportingController extends Controller
{
    private ReportingService $reportingService;

    public function __construct(ReportingService $reportingService)
    {
        $this->reportingService = $reportingService;
    }

    /**
     * Display the reporting dashboard
     */
    public function index(Request $request)
    {
        $window = $request->get('window', 'daily');
        $startDate = $request->filled('start_date') ? Carbon::parse($request->start_date) : null;
        $endDate = $request->filled('end_date') ? Carbon::parse($request->end_date) : null;

        $report = $this->reportingService->generateKPIReport($window, $startDate, $endDate);
        $topSellers = $this->reportingService->getTopSellingDishes($window, 10, $startDate, $endDate);
        
        // Calculate additional metrics
        $grossMargin = $this->reportingService->calculateGrossMargin(
            $report['sales_metrics']['net_sales'],
            $report['inventory_metrics']['cogs']
        );

        return view('admin.reporting.index', compact('report', 'topSellers', 'grossMargin', 'window'));
    }

    /**
     * API endpoint for KPI data
     */
    public function api(Request $request)
    {
        $request->validate([
            'window' => 'required|in:daily,weekly,monthly',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $window = $request->window;
        $startDate = $request->filled('start_date') ? Carbon::parse($request->start_date) : null;
        $endDate = $request->filled('end_date') ? Carbon::parse($request->end_date) : null;

        $report = $this->reportingService->generateKPIReport($window, $startDate, $endDate);
        $topSellers = $this->reportingService->getTopSellingDishes($window, 10, $startDate, $endDate);
        
        $grossMargin = $this->reportingService->calculateGrossMargin(
            $report['sales_metrics']['net_sales'],
            $report['inventory_metrics']['cogs']
        );

        return response()->json([
            'success' => true,
            'data' => [
                'report' => $report,
                'top_sellers' => $topSellers,
                'gross_margin_percent' => round($grossMargin, 2),
            ]
        ]);
    }

    /**
     * Export report data as CSV
     */
    public function export(Request $request)
    {
        $request->validate([
            'window' => 'required|in:daily,weekly,monthly',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $window = $request->window;
        $startDate = $request->filled('start_date') ? Carbon::parse($request->start_date) : null;
        $endDate = $request->filled('end_date') ? Carbon::parse($request->end_date) : null;

        $report = $this->reportingService->generateKPIReport($window, $startDate, $endDate);
        $topSellers = $this->reportingService->getTopSellingDishes($window, 50, $startDate, $endDate);

        $filename = "kpi_report_{$window}_" . now()->format('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0'
        ];

        return response()->stream(function () use ($report, $topSellers) {
            $handle = fopen('php://output', 'w');
            
            // Report summary
            fputcsv($handle, ['KPI Report Summary']);
            fputcsv($handle, ['Period', $report['period']['start_date'] . ' to ' . $report['period']['end_date']]);
            fputcsv($handle, ['Generated At', $report['generated_at']]);
            fputcsv($handle, []);
            
            // Sales metrics
            fputcsv($handle, ['Sales Metrics']);
            fputcsv($handle, ['Metric', 'Value']);
            foreach ($report['sales_metrics'] as $key => $value) {
                fputcsv($handle, [ucwords(str_replace('_', ' ', $key)), $value]);
            }
            fputcsv($handle, []);
            
            // Inventory metrics
            fputcsv($handle, ['Inventory Metrics']);
            fputcsv($handle, ['Metric', 'Value']);
            foreach ($report['inventory_metrics'] as $key => $value) {
                fputcsv($handle, [ucwords(str_replace('_', ' ', $key)), $value]);
            }
            fputcsv($handle, []);
            
            // Operational metrics
            fputcsv($handle, ['Operational Metrics']);
            fputcsv($handle, ['Metric', 'Value']);
            foreach ($report['operational_metrics'] as $key => $value) {
                fputcsv($handle, [ucwords(str_replace('_', ' ', $key)), $value]);
            }
            fputcsv($handle, []);
            
            // Top sellers
            fputcsv($handle, ['Top Selling Dishes']);
            fputcsv($handle, ['Rank', 'Dish Name', 'Units Sold', 'Revenue']);
            foreach ($topSellers as $index => $seller) {
                fputcsv($handle, [
                    $index + 1,
                    $seller['dish_name'],
                    $seller['total_sold'],
                    $seller['revenue']
                ]);
            }
            
            fclose($handle);
        }, 200, $headers);
    }

    /**
     * Clear reporting cache
     */
    public function clearCache()
    {
        $this->reportingService->clearCache();
        
        return response()->json([
            'success' => true,
            'message' => 'Reporting cache cleared successfully'
        ]);
    }
}
