<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\Order;

class MeKpiController extends Controller
{
    public function __invoke(Request $request)
    {
        $userId = Auth::id();
        if (!$userId) {
            return response()->json(['ok' => false, 'error' => 'unauthenticated'], 401);
        }

        $cacheKey = "me:kpis:v1:user:$userId";

        try {
            $data = Cache::remember($cacheKey, 30, function () use ($userId) {
                $totalOrders = Order::where('user_id', $userId)->count();

                $active = Order::where('user_id', $userId)
                    ->whereNotIn('status', ['Delivered', 'Cancelled'])
                    ->latest()
                    ->first();

                // Avg delivery minutes for completed orders (best-effort)
                $avgMinutes = null;
                try {
                    $avgMinutes = DB::table('orders')
                        ->where('user_id', $userId)
                        ->where('status', 'Delivered')
                        ->selectRaw('AVG(TIMESTAMPDIFF(MINUTE, created_at, updated_at)) as avg_minutes')
                        ->value('avg_minutes');
                    if ($avgMinutes !== null) {
                        $avgMinutes = (int) round((float) $avgMinutes);
                    }
                } catch (\Throwable $e) {
                    $avgMinutes = null;
                }

                $addressesCount = 0;
                if (Schema::hasTable('customer_addresses')) {
                    try {
                        $addressesCount = DB::table('customer_addresses')->where('user_id', $userId)->count();
                    } catch (\Throwable $e) {
                        $addressesCount = 0;
                    }
                }

                // Favorites count (distinct dishes ever ordered) – best-effort
                $favoritesCount = 0;
                try {
                    if (Schema::hasTable('order_items')) {
                        $favoritesCount = DB::table('order_items')
                            ->join('orders', 'order_items.order_id', '=', 'orders.id')
                            ->where('orders.user_id', $userId)
                            ->distinct('order_items.dish_id')
                            ->count('order_items.dish_id');
                    }
                } catch (\Throwable $e) {
                    $favoritesCount = 0;
                }

                // ETA for active order: created_at + configured minutes (fallback 30)
                $etaIso = null; $etaMinutes = null; $activeOrder = null;
                if ($active) {
                    $avgConfigured = (int) (config('app.avg_delivery_minutes', 30));
                    $eta = optional($active->created_at)->copy()->addMinutes($avgConfigured);
                    $etaIso = $eta ? $eta->toIso8601String() : null;
                    $etaMinutes = $avgConfigured;
                    $activeOrder = [
                        'id' => $active->id,
                        'status' => $active->status ?? $active->display_status ?? 'Received',
                        'created_at' => optional($active->created_at)->toIso8601String(),
                        'eta' => $etaIso,
                        'eta_minutes' => $etaMinutes,
                    ];
                }

                return [
                    'ok' => true,
                    'db_unavailable' => false,
                    'total_orders' => $totalOrders,
                    'avg_delivery_minutes' => $avgMinutes,
                    'addresses_count' => $addressesCount,
                    'favorites_count' => $favoritesCount,
                    'active_order' => $activeOrder,
                ];
            });

            return response()->json($data);
        } catch (\Throwable $e) {
            return response()->json([
                'ok' => false,
                'db_unavailable' => true,
                'total_orders' => 0,
                'avg_delivery_minutes' => null,
                'addresses_count' => 0,
                'favorites_count' => 0,
                'active_order' => null,
            ], 200);
        }
    }
}
