<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        return view('auth.register');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        try {
            $validated = $request->validate([
                'first_name' => ['required','string','max:100'],
                'last_name' => ['required','string','max:100'],
                'gender' => ['nullable','in:male,female'],
                'email' => ['required','string','lowercase','email','max:100','unique:'.User::class],
                'phone' => ['required','string','max:20','unique:users,phone'],
                'room_number' => ['nullable','string','max:10'],
                'password' => ['required','confirmed', Rules\Password::defaults()],
            ]);

            $fullName = trim($validated['first_name'].' '.$validated['last_name']);

            $user = User::create([
                'name' => $fullName,
                'first_name' => $validated['first_name'],
                'last_name' => $validated['last_name'],
                'gender' => $validated['gender'] ?? null,
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'room_number' => $validated['room_number'] ?? null,
                'password' => Hash::make($validated['password']),
                'role' => 'customer',
                'is_guest' => false,
            ]);

            event(new Registered($user));

            Auth::login($user);

            // Assign default Customer role via Spatie
            try { $user->assignRole('Customer'); } catch (\Throwable $e) { /* roles may not be seeded yet */ }

            // Role-based redirect
            return redirect()->intended($this->redirectFor($user));
        } catch (\Illuminate\Validation\ValidationException $ve) {
            throw $ve;
        } catch (\Throwable $e) {
            return back()
                ->withErrors(['email' => __('Registration is temporarily unavailable. Please try again shortly.')])
                ->withInput($request->except(['password','password_confirmation']));
        }
    }

    protected function redirectFor(User $user): string
    {
        if ($user->hasAnyRole(['Super Admin','Manager','Kitchen','Cashier','Content Editor','Rider','Support'])) {
            return route('admin.dashboard');
        }
        return route('dashboard');
    }
}
