<?php

namespace App\Http\Controllers;

use App\Models\Dish;
use Illuminate\Http\Request;

class CartController extends Controller
{
    protected function cartCount(array $cart): int
    {
        $sum = 0;
        foreach ($cart['items'] as $qty) {
            $sum += is_array($qty) && isset($qty['qty']) ? (int)$qty['qty'] : (int)$qty;
        }
        return $sum;
    }
    protected function getCart(): array
    {
        return session('cart', ['items' => []]);
    }

    protected function saveCart(array $cart): void
    {
        session(['cart' => $cart]);
    }

    public function index()
    {
        $dbUnavailable = false;
        $cart = $this->getCart();
        $subtotal = 0;
        $lines = [];
        $belowMin = false; $minSubtotal = 0; $atCapacity = false;
        try {
            $dishIds = array_keys($cart['items']);
            $dishes = Dish::whereIn('id', $dishIds)->get()->keyBy('id');
            foreach ($cart['items'] as $dishId => $qty) {
                $dish = $dishes[$dishId] ?? null;
                if (!$dish) continue;
                $lineTotal = (float) $dish->price * (int) $qty;
                $subtotal += $lineTotal;
                $lines[] = compact('dish','qty','lineTotal');
            }
            // Settings-driven banners
            $minSubtotal = (float) \App\Models\Setting::get('order.min_subtotal', 0);
            $belowMin = $subtotal < $minSubtotal;
            $maxOpen = (int) \App\Models\Setting::get('capacity.max_open_orders', 50);
            $openCount = \App\Models\Order::openOrdersCount();
            $atCapacity = $openCount >= $maxOpen;
        } catch (\Throwable $e) {
            $dbUnavailable = true;
            // leave defaults; view will show a friendly notice
        }
        return view('cart.index', compact('lines','subtotal','belowMin','minSubtotal','atCapacity','dbUnavailable'));
    }

    public function add(Request $request)
    {
        try {
            $data = $request->validate([
                'dish_id' => 'required|exists:dishes,id',
                'quantity' => 'nullable|integer|min:1|max:100',
            ]);
            $qty = $data['quantity'] ?? 1;
            $cart = $this->getCart();
            $cart['items'][$data['dish_id']] = ($cart['items'][$data['dish_id']] ?? 0) + $qty;
            $this->saveCart($cart);
            if ($request->expectsJson()) {
                return response()->json([
                    'ok' => true,
                    'message' => 'Added to cart',
                    'count' => $this->cartCount($cart),
                ]);
            }
            return back()->with('status','Added to cart');
        } catch (\Throwable $e) {
            if ($request->expectsJson()) {
                return response()->json(['ok' => false, 'message' => 'Temporarily unavailable. Please try again shortly.'], 200);
            }
            return back()->with('status','Temporarily unavailable. Please try again shortly.');
        }
    }

    public function update(Request $request)
    {
        try {
            $data = $request->validate([
                'dish_id' => 'required|exists:dishes,id',
                'quantity' => 'required|integer|min:1|max:100',
            ]);
            $cart = $this->getCart();
            $cart['items'][$data['dish_id']] = $data['quantity'];
            $this->saveCart($cart);
            if ($request->expectsJson()) {
                return response()->json([
                    'ok' => true,
                    'message' => 'Cart updated',
                    'count' => $this->cartCount($cart),
                ]);
            }
            return back()->with('status','Cart updated');
        } catch (\Throwable $e) {
            if ($request->expectsJson()) {
                return response()->json(['ok' => false, 'message' => 'Temporarily unavailable. Please try again.'], 200);
            }
            return back()->with('status','Temporarily unavailable. Please try again.');
        }
    }

    public function remove(Request $request)
    {
        try {
            $data = $request->validate(['dish_id' => 'required|exists:dishes,id']);
            $cart = $this->getCart();
            unset($cart['items'][$data['dish_id']]);
            $this->saveCart($cart);
            if ($request->expectsJson()) {
                return response()->json([
                    'ok' => true,
                    'message' => 'Removed',
                    'count' => $this->cartCount($cart),
                ]);
            }
            return back()->with('status','Removed');
        } catch (\Throwable $e) {
            if ($request->expectsJson()) {
                return response()->json(['ok' => false, 'message' => 'Temporarily unavailable. Please try again.'], 200);
            }
            return back()->with('status','Temporarily unavailable. Please try again.');
        }
    }

    public function clear(Request $request)
    {
        try {
            $this->saveCart(['items' => []]);
            if ($request->expectsJson()) {
                return response()->json([
                    'ok' => true,
                    'message' => 'Cart cleared',
                    'count' => 0,
                ]);
            }
            return back()->with('status','Cart cleared');
        } catch (\Throwable $e) {
            if ($request->expectsJson()) {
                return response()->json(['ok' => false, 'message' => 'Temporarily unavailable. Please try again.'], 200);
            }
            return back()->with('status','Temporarily unavailable. Please try again.');
        }
    }
}
