<?php

namespace App\Http\Controllers;

use App\Models\Dish;
use App\Models\Order;
use App\Models\OrderItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth as AuthFacade;
use App\Models\Setting;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Schema;
use App\Models\Stay;
use App\Services\Delivery\{DeliveryContext, DeliveryVerificationService};
use App\Services\Inventory\InventoryService;
use App\Support\Delivery\HotelSettings;
use App\Support\Geo\Distance;
use App\Events\OrderDeliveryEvaluated;
use App\Models\CustomerAddress;

class CheckoutController extends Controller
{
    protected function computeDeliveryFee(float $subtotal): float
    {
        try {
            $base = (float) Setting::get('delivery.fee_base', 1000);
        } catch (\Throwable $e) {
            $base = 1000;
        }
        try {
            $freeThreshold = (float) Setting::get('delivery.free_threshold', 20000);
        } catch (\Throwable $e) {
            $freeThreshold = 20000;
        }
        return $subtotal >= $freeThreshold ? 0.0 : $base;
    }

    protected function computeGuestDiscount(float $subtotal): float
    {
        // Assumption: hotel guests get 10% discount on subtotal
        $user = Auth::user();
        if ($user && $user->is_guest) {
            return round($subtotal * 0.10, 2);
        }
        return 0.0;
    }

    protected function computeServiceCharge(float $subtotal): float
    {
        try {
            $percent = (float) Setting::get('order.service_charge_percent', 0);
        } catch (\Throwable $e) {
            $percent = 0;
        }
        return round($subtotal * ($percent / 100), 2);
    }

    public function index()
    {
        $cart = session('cart', ['items' => []]);
        if (empty($cart['items'])) {
            return redirect()->route('cart.index')->with('status','Your cart is empty');
        }
        $dbUnavailable = false;
        $subtotal = 0; $delivery = 0; $discount = 0; $service = 0; $total = 0; $minSubtotal = 0; $belowMin = false; $atCapacity = false; $roomServiceEnabled = true;
        try {
            $dishIds = array_keys($cart['items']);
            $dishes = Dish::whereIn('id', $dishIds)->get()->keyBy('id');
            foreach ($cart['items'] as $dishId => $qty) {
                $dish = $dishes[$dishId] ?? null;
                if ($dish) { $subtotal += (float)$dish->price * (int)$qty; }
            }
            // capacity guard (soft warning handled in store)
            try { $maxOpen = (int) Setting::get('capacity.max_open_orders', 50); } catch (\Throwable $e) { $maxOpen = 50; }
            try { $openCount = \App\Models\Order::openOrdersCount(); } catch (\Throwable $e) { $openCount = 0; }
            $atCapacity = $openCount >= $maxOpen;

            $delivery = $this->computeDeliveryFee($subtotal);
            $discount = $this->computeGuestDiscount($subtotal);
            $service = $this->computeServiceCharge($subtotal);
            $total = max(0, $subtotal + $delivery + $service - $discount);
            try { $minSubtotal = (float) Setting::get('order.min_subtotal', 0); } catch (\Throwable $e) { $minSubtotal = 0; }
            $belowMin = $subtotal < $minSubtotal;
            try { $roomServiceEnabled = (bool) Setting::get('room_service.enabled', true); } catch (\Throwable $e) { $roomServiceEnabled = true; }
        } catch (\Throwable $e) {
            $dbUnavailable = true;
            // Keep safe defaults and show banner; form will be disabled in the view
            $subtotal = 0; $delivery = $this->computeDeliveryFee(0); $discount = 0; $service = $this->computeServiceCharge(0); $total = 0; $minSubtotal = 0; $belowMin = false; $atCapacity = true; $roomServiceEnabled = false;
        }
        return view('checkout.index', compact('subtotal','delivery','discount','service','total','belowMin','minSubtotal','atCapacity','roomServiceEnabled','dbUnavailable'));
    }

    public function store(Request $request, DeliveryVerificationService $verifier)
    {
        // Early DB connectivity check to fail fast when offline
        try { DB::select('SELECT 1'); } catch (\Throwable $e) {
            return back()->withErrors(['checkout' => 'Ordering is temporarily unavailable. Please try again shortly.'])->withInput();
        }

        $user = Auth::user();
        $isGuest = $user && ($user->is_guest ?? false);
        $deliveryMode = $isGuest ? ($request->input('delivery_mode', 'room')) : 'external';
        $roomNumber = trim($request->input('room_number', $user->room_number ?? ''));
        $address = trim($request->input('delivery_address', ''));

        // Validation rules
        $rules = [
            'payment_method' => 'required|string|in:Cash on Delivery,Paystack,Charge to Room',
            'tip' => 'nullable|numeric|min:0|max:500000',
            'address_id' => 'nullable|integer|exists:customer_addresses,id',
        ];
        if ($isGuest) {
            $rules['delivery_mode'] = 'required|in:room,external';
            if ($deliveryMode === 'room') {
                $rules['room_number'] = 'required|string|max:20';
            } else {
                $rules['delivery_address'] = 'required|string|min:3|max:255';
            }
        } else {
            $rules['delivery_address'] = 'required|string|min:3|max:255';
        }
        $data = $request->validate($rules);

        // Large hotel: validate room number against active stays
        try {
            $largeHotel = (int)\App\Models\Stay::active()->count() > 50; // threshold can be adjusted
        } catch (\Throwable $e) {
            $largeHotel = false;
        }
        if ($isGuest && $deliveryMode === 'room') {
            if ($largeHotel) {
                $validRooms = \App\Models\Stay::activeRoomNumbers();
                if (!in_array(strtolower($roomNumber), $validRooms)) {
                    return back()->withErrors(['room_number' => 'Room not found or not checked in.'])->withInput();
                }
            }
            $address = 'Room ' . $roomNumber;
        }
        if (!$isGuest || ($isGuest && $deliveryMode === 'external')) {
            // Must look like a street address (at least one number and one space)
            if (!preg_match('/\d+\s+\S+/', $address)) {
                return back()->withErrors(['delivery_address' => 'Please enter a valid delivery address (e.g., 123 Main St).'])->withInput();
            }
        }

    $cart = session('cart', ['items' => []]);
        if (empty($cart['items'])) {
            return redirect()->route('cart.index')->with('status','Your cart is empty');
        }

        $dishIds = array_keys($cart['items']);
        try {
            $dishes = Dish::whereIn('id', $dishIds)->get()->keyBy('id');
        } catch (\Throwable $e) {
            return back()->withErrors(['checkout' => 'Ordering is temporarily unavailable. Please try again shortly.'])->withInput();
        }

    DB::beginTransaction();
    try {
            $subtotal = 0; 
            foreach ($cart['items'] as $dishId => $qty) {
                $dish = $dishes[$dishId] ?? null;
                if (!$dish) continue;
                // Check stock availability
                $q = (int) $qty;
                if (Schema::hasColumn('dishes', 'stock') && isset($dish->stock) && $dish->stock !== null && $dish->stock < $q) {
                    DB::rollBack();
                    return back()->withErrors(['checkout' => "Not enough stock for {$dish->name}. Available: {$dish->stock}"])->withInput();
                }
                $subtotal += (float)$dish->price * $q;
            }
            // Enforce minimum subtotal (if configured)
            try { $minSubtotal = (float) Setting::get('order.min_subtotal', 0); } catch (\Throwable $e) { $minSubtotal = 0; }
            if ($minSubtotal > 0 && $subtotal < $minSubtotal) {
                DB::rollBack();
                return back()->withErrors([
                    'checkout' => 'Minimum order subtotal is ₦'.number_format($minSubtotal).'.'
                ])->withInput();
            }

            // Capacity guard (open orders)
            try { $maxOpen = (int) Setting::get('capacity.max_open_orders', 50); } catch (\Throwable $e) { $maxOpen = PHP_INT_MAX; }
            try { $openOrders = \App\Models\Order::openOrdersCount(); } catch (\Throwable $e) { $openOrders = 0; }
            if ($openOrders >= $maxOpen) {
                DB::rollBack();
                return back()->withErrors(['checkout' => 'We are currently at capacity. Please try again shortly.'])->withInput();
            }

            // Room service toggle
            $roomEnabled = true; try { $roomEnabled = (bool) Setting::get('room_service.enabled', true); } catch (\Throwable $e) { $roomEnabled = false; }
            if (($data['payment_method'] ?? '') === 'Charge to Room' && ! $roomEnabled) {
                DB::rollBack();
                return back()->withErrors(['checkout' => 'Room service is currently unavailable.'])->withInput();
            }

            $delivery = $this->computeDeliveryFee($subtotal);
            $discount = $this->computeGuestDiscount($subtotal);
            $service = $this->computeServiceCharge($subtotal);
            $tip = (float) ($data['tip'] ?? 0);
            $total = max(0, $subtotal + $delivery + $service - $discount + $tip);

            // Determine active stay (simple heuristic)
            $user = Auth::user();
            $stay = null;
            if ($user) {
                $stay = Stay::query()
                    ->where('user_id', $user->id)
                    ->where(function($q){
                        $now = now();
                        $q->where('check_in_at','<=',$now)->where('check_out_at','>=',$now);
                    })
                    ->orderByDesc('check_in_at')
                    ->first();
            }

            // Decide delivery mode heuristically:
            $rawAddress = $data['delivery_address'] ?? null;
            $roomIntent = ($data['payment_method'] === 'Charge to Room')
                || !empty($data['room_number'])
                || ($rawAddress && preg_match('/^room\s+/i', $rawAddress));
            $deliveryMode = $roomIntent ? 'room' : 'external';

            // Enforce non-empty destination
            if ($deliveryMode === 'room') {
                $roomNumber = $data['room_number'] ?? ($stay?->room_number);
                if ($roomNumber) {
                    // Sanitize room number (alnum + dash only)
                    $roomNumber = preg_replace('/[^A-Za-z0-9\-]/','',$roomNumber);
                }
                if (!$roomNumber && $rawAddress && preg_match('/room\s+(\S+)/i', $rawAddress, $m)) {
                    $roomNumber = $m[1];
                }
                if (!$roomNumber) {
                    DB::rollBack();
                    return back()->withErrors(['delivery_address' => 'Room number required for in-house delivery.'])->withInput();
                }
                // Normalize raw address representation
                $rawAddress = 'Room '.$roomNumber;
                $data['room_number'] = $roomNumber;
            } else {
                if (empty($rawAddress) && empty($data['address_id'])) {
                    DB::rollBack();
                    return back()->withErrors(['delivery_address' => 'External delivery address required.'])->withInput();
                }
            }

            $ctx = new DeliveryContext(
                user: $user,
                activeStay: $stay,
                deliveryMode: $deliveryMode,
                rawAddress: $deliveryMode === 'external' ? $rawAddress : null,
                radiusKm: HotelSettings::radiusKm(),
                hotelLat: HotelSettings::latitude(),
                hotelLng: HotelSettings::longitude(),
                requireVerification: (bool) config('delivery.require_verification')
            );
            $decision = $verifier->decide($ctx);

            $computedDistance = Distance::km($decision->lat, $decision->lng, HotelSettings::latitude(), HotelSettings::longitude());
            if ($decision->distanceKm === null && $computedDistance !== null) {
                // We treat this as the distance for now (future phases may re-evaluate)
                $decisionDistance = $computedDistance;
            } else {
                $decisionDistance = $decision->distanceKm;
            }

            // Feature flag scaffolding: if a future decision is FLAG/REJECT and verification required
            if (config('delivery.require_verification')) {
                if ($decision->status === 'REJECT') {
                    DB::rollBack();
                    return back()->withErrors(['checkout' => 'Address not serviceable.'])->withInput();
                }
                if ($decision->status === 'FLAG') {
                    $initialState = \App\Enums\OrderState::AWAITING_REVIEW->value;
                } else {
                    $initialState = ($data['payment_method'] === 'Paystack'
                        ? \App\Enums\OrderState::AWAITING_PAYMENT
                        : \App\Enums\OrderState::AWAITING_KITCHEN)->value;
                }
            } else {
                $initialState = ($data['payment_method'] === 'Paystack'
                    ? \App\Enums\OrderState::AWAITING_PAYMENT
                    : \App\Enums\OrderState::AWAITING_KITCHEN)->value;
            }

            $customerAddressId = null;
            if ($deliveryMode === 'external' && $rawAddress) {
                $normalized = preg_replace('/\s+/', ' ', trim(mb_strtolower($rawAddress)));
                $hash = sha1($normalized);
                $addr = CustomerAddress::firstOrNew([
                    'user_id' => Auth::id(),
                    'address_hash' => $hash,
                ]);
                if (! $addr->exists) {
                    $addr->raw_address = $rawAddress;
                    $addr->formatted_address = $decision->meta['formatted'] ?? $rawAddress;
                    $addr->latitude = $decision->lat;
                    $addr->longitude = $decision->lng;
                    $addr->times_used = 1;
                    $addr->last_used_at = now();
                    // First address becomes primary automatically if none exist
                    $hasAny = CustomerAddress::where('user_id', Auth::id())->exists();
                    $addr->is_primary = ! $hasAny;
                    $addr->save();
                } else {
                    $addr->times_used = ($addr->times_used ?? 0) + 1;
                    $addr->last_used_at = now();
                    // Update geo if previously empty and now available
                    if (!$addr->latitude && $decision->lat) $addr->latitude = $decision->lat;
                    if (!$addr->longitude && $decision->lng) $addr->longitude = $decision->lng;
                    $addr->save();
                }
                $customerAddressId = $addr->id;
            }

            $order = Order::create([
                'user_id' => Auth::id(),
                'stay_id' => $stay?->id,
                'customer_address_id' => $customerAddressId,
                'subtotal' => $subtotal,
                'delivery_fee' => $delivery,
                'discount' => $discount,
                'service_charge' => $service,
                'total' => $total,
                'status' => 'Received',
                'state' => $initialState,
                'delivery_mode' => $deliveryMode,
                'ordered_by_guest' => (bool) $stay,
                'guest_room_number' => $stay?->room_number,
                'delivery_address' => $rawAddress,
                'delivery_address_verified' => $decision->status === 'ACCEPT',
                'delivery_latitude' => $decision->lat,
                'delivery_longitude' => $decision->lng,
                'delivery_distance_km' => $decisionDistance,
                'verification_log' => $decision->meta ? json_encode($decision->meta) : null,
                'payment_method' => $data['payment_method'],
                'tip' => $tip,
            ]);

            event(new OrderDeliveryEvaluated($order, $decision->meta + ['status'=>$decision->status]));
            \Log::info('order.delivery.created', [
                'order_id' => $order->id,
                'mode' => $order->delivery_mode,
                'state' => $order->state,
                'verified' => $order->delivery_address_verified,
                'distance_km' => $order->delivery_distance_km,
                'phase' => $decision->meta['phase'] ?? null,
            ]);

            foreach ($cart['items'] as $dishId => $qty) {
                $dish = $dishes[$dishId] ?? null;
                if (!$dish) continue;
                OrderItem::create([
                    'order_id' => $order->id,
                    'dish_id' => $dish->id,
                    'quantity' => (int)$qty,
                    'price' => (float)$dish->price,
                ]);
                
                // Use InventoryService for stock management
                if (Schema::hasColumn('dishes', 'stock') && isset($dish->stock)) {
                    $inventoryService = app(InventoryService::class);
                    try {
                        $inventoryService->recordSale(
                            dish: $dish,
                            quantity: (int)$qty,
                            order: $order,
                            user: AuthFacade::user(),
                            note: "Order #{$order->id} checkout"
                        );
                    } catch (\Illuminate\Validation\ValidationException $e) {
                        // This should not happen due to pre-checkout validation, but handle gracefully
                        DB::rollBack();
                        return back()->withErrors(['checkout' => $e->getMessage()])->withInput();
                    }
                }
            }

            // If Paystack, initialize transaction and redirect to payment page
            if ($data['payment_method'] === 'Paystack') {
                $secret = config('services.paystack.secret') ?? env('PAYSTACK_SECRET');
                if (!$secret) {
                    DB::rollBack();
                    return back()->withErrors(['checkout' => 'Paystack not configured. Set PAYSTACK_SECRET in .env.'])->withInput();
                }

                $reference = 'BD'.now()->format('YmdHis').$order->id;
                $callback = route('paystack.callback');
                // If the app is running in a subdirectory (APP_URL path), ensure callback is absolute and correct
                // route() already produces absolute URL using APP_URL; kept for clarity.

                $response = Http::withToken($secret)
                    ->acceptJson()
                    ->post('https://api.paystack.co/transaction/initialize', [
                        'email' => Auth::user()->email,
                        'amount' => (int) round($total * 100), // kobo
                        'reference' => $reference,
                        'callback_url' => $callback,
                        'metadata' => [
                            'order_id' => $order->id,
                            'user_id' => Auth::id(),
                        ],
                    ]);

                if (!$response->ok() || !data_get($response->json(), 'status')) {
                    \Log::warning('Paystack initialize failed', [
                        'http_status' => $response->status(),
                        'body' => $response->body(),
                        'order_id' => $order->id,
                    ]);
                    DB::rollBack();
                    $msg = data_get($response->json(), 'message') ?: 'Failed to initialize Paystack.';
                    return back()->withErrors(['checkout' => $msg])->withInput();
                }

                $order->payment_reference = $reference;
                $order->save();
                DB::commit();
                // Don't clear cart until payment completes
                $authUrl = data_get($response->json(), 'data.authorization_url');
                return redirect()->away($authUrl);
            }

            DB::commit();
            session()->forget('cart');
            return redirect()->route('orders.show', $order)->with('status','Order placed');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->withErrors(['checkout' => 'Failed to place order.'])->withInput();
        }
    }
}
