<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use App\Models\Order;
use App\Models\Dish;
use App\Models\CustomerAddress;

class CustomerDashboardController extends Controller
{
    public function __invoke(Request $request)
    {
        $user = Auth::user();
        $dbUnavailable = false;
        try {
            // Customer's personal order data only
            $recentOrders = Order::where('user_id', $user->id)
                ->with(['items.dish'])
                ->latest()
                ->limit(5)
                ->get();
        } catch (\Throwable $e) {
            $dbUnavailable = true;
            $recentOrders = collect();
        }

        // Customer's favorite dishes based on order history
        $favoriteDishes = $this->getFavoriteDishes($user->id);

        // Customer's current active order (if any)
        try {
            $activeOrder = Order::where('user_id', $user->id)
                ->whereNotIn('status', ['Delivered', 'Cancelled'])
                ->latest()
                ->first();
        } catch (\Throwable $e) {
            $dbUnavailable = true;
            $activeOrder = null;
        }

        // Customer's saved addresses (handle table existence gracefully)
        $savedAddresses = collect([]);
        try {
            if (\Schema::hasTable('customer_addresses')) {
                $savedAddresses = CustomerAddress::where('user_id', $user->id)
                    ->orderBy('is_primary', 'desc')
                    ->limit(3)
                    ->get();
            }
        } catch (\Throwable $e) {
            $dbUnavailable = true;
            $savedAddresses = collect([]);
        }

        // Personal stats (customer's own data only)
        try {
            $personalStats = [
                'total_orders' => Order::where('user_id', $user->id)->count(),
                'favorite_dishes_count' => $favoriteDishes->count(),
                'saved_addresses_count' => \Schema::hasTable('customer_addresses')
                    ? CustomerAddress::where('user_id', $user->id)->count()
                    : 0,
            ];
        } catch (\Throwable $e) {
            $dbUnavailable = true;
            $personalStats = [
                'total_orders' => 0,
                'favorite_dishes_count' => 0,
                'saved_addresses_count' => 0,
            ];
        }

        return view('dashboard', compact(
            'user',
            'recentOrders', 
            'favoriteDishes',
            'activeOrder',
            'savedAddresses',
            'personalStats',
            'dbUnavailable'
        ));
    }
    
    /**
     * Get customer's favorite dishes based on order frequency
     */
    private function getFavoriteDishes($userId)
    {
        try {
            if (!Order::where('user_id', $userId)->exists()) {
                // If no orders, return a few popular dishes
                return Dish::limit(6)->get();
            }
            
            return Dish::select('dishes.*')
                ->join('order_items', 'dishes.id', '=', 'order_items.dish_id')
                ->join('orders', 'order_items.order_id', '=', 'orders.id')
                ->where('orders.user_id', $userId)
                ->groupBy('dishes.id', 'dishes.name', 'dishes.price', 'dishes.image', 'dishes.description', 'dishes.category_id', 'dishes.created_at', 'dishes.updated_at', 'dishes.deleted_at')
                ->orderByRaw('COUNT(order_items.id) DESC')
                ->limit(6)
                ->get();
        } catch (\Exception $e) {
            // Gracefully handle any database issues
            return collect([]);
        }
    }
}