<?php

namespace App\Http\Controllers;

use App\Models\Order;
use Illuminate\Support\Facades\Auth;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Http\Request;

class CustomerOrderController extends Controller
{
    public function index(Request $request)
    {
        $dbUnavailable = false;
        $currentTab = strtolower($request->query('status', 'all'));
        $q = trim((string) $request->query('q', ''));

        try {
            $query = Order::query()->where('user_id', Auth::id());

            // Status filter tabs
            switch ($currentTab) {
                case 'active':
                    $query->whereNotIn('status', ['Delivered', 'Cancelled']);
                    break;
                case 'completed':
                case 'delivered':
                    $query->where('status', 'Delivered');
                    $currentTab = 'completed';
                    break;
                case 'cancelled':
                    $query->where('status', 'Cancelled');
                    break;
                default:
                    $currentTab = 'all';
            }

            // Simple search: numeric ID
            if ($q !== '') {
                if (ctype_digit($q)) {
                    $query->where('id', (int) $q);
                }
            }

            $orders = $query->latest()->paginate(20)->appends($request->query());
        } catch (\Throwable $e) {
            $dbUnavailable = true;
            $orders = new LengthAwarePaginator([], 0, 20, 1, [
                'path' => request()->url(),
                'query' => request()->query(),
            ]);
        }
        return view('orders.index', compact('orders','dbUnavailable','currentTab','q'));
    }

    public function show(Order $order)
    {
        abort_unless($order->user_id === Auth::id(), 403);
        try {
            $order->load(['items.dish']);
        } catch (\Throwable $e) {
            // Proceed to view; relationships may be empty
        }
        return view('orders.show', compact('order'));
    }

    /**
     * Lightweight JSON for polling an order's status.
     * GET /orders/{order}/status.json
     */
    public function statusJson(Order $order)
    {
        abort_unless($order->user_id === Auth::id(), 403);

        try {
            // Best-effort fresh state; if relationships fail, ignore
            $order->refresh();
            $avgConfigured = (int) (config('app.avg_delivery_minutes', 30));
            $eta = optional($order->created_at)->copy()->addMinutes($avgConfigured);

            return response()->json([
                'ok' => true,
                'db_unavailable' => false,
                'id' => $order->id,
                'status' => $order->status ?? $order->display_status ?? 'Received',
                'created_at' => optional($order->created_at)->toIso8601String(),
                'updated_at' => optional($order->updated_at)->toIso8601String(),
                'eta' => $eta ? $eta->toIso8601String() : null,
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'ok' => false,
                'db_unavailable' => true,
                'id' => $order->id,
                'status' => $order->status ?? 'Received',
                'created_at' => optional($order->created_at)->toIso8601String(),
                'updated_at' => optional($order->updated_at)->toIso8601String(),
                'eta' => null,
            ], 200);
        }
    }
}
