<?php

namespace App\Http\Controllers;

use App\Models\Dish;
use App\Models\Category;
use App\Models\InventoryMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Schema;

class DishController extends Controller
{
    public function index(Request $request)
    {
        $q = trim((string) $request->get('q', ''));
        $categoryId = $request->get('category_id');
        $perPage = (int) ($request->get('perPage', 15));
        if (!in_array($perPage, [10,15,25,50,100], true)) { $perPage = 15; }

        $showTrashed = (bool)$request->get('trashed');

        $sort = (string) $request->get('sort', 'updated');
        $dir = strtolower((string) $request->get('dir', 'desc')) === 'asc' ? 'asc' : 'desc';
        $sortable = [
            'name' => 'dishes.name',
            'price' => 'dishes.price',
            'category' => 'categories.name',
            'updated' => 'dishes.updated_at',
        ];
        $orderBy = $sortable[$sort] ?? $sortable['updated'];

        $query = Dish::query()
            ->leftJoin('categories', 'dishes.category_id', '=', 'categories.id')
            ->select('dishes.*')
            ->addSelect([
                // Subselect to fetch last movement timestamp per dish (avoids N+1 in view)
                'last_movement_at' => InventoryMovement::selectRaw('MAX(created_at)')
                    ->whereColumn('inventory_movements.dish_id', 'dishes.id')
            ])
            ->with('category');

        if ($showTrashed) {
            $query->withTrashed();
        }

        if ($q !== '') {
            $query->where(function($w) use ($q) {
                $w->where('dishes.name', 'like', "%$q%")
                  ->orWhere('dishes.description', 'like', "%$q%")
                  ->orWhere('categories.name', 'like', "%$q%");
            });
        }

        if ($categoryId) {
            $query->where('dishes.category_id', $categoryId);
        }

        $dishes = $query->orderBy($orderBy, $dir)->paginate($perPage)->withQueryString();
        $categories = Category::orderBy('name')->get(['id','name']);
        return view('admin.dishes.index', compact('dishes', 'categories', 'q', 'sort', 'dir', 'perPage', 'categoryId','showTrashed'));
    }

    public function create()
    {
        $categories = Category::orderBy('name')->get();
        return view('admin.dishes.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'stock' => 'nullable|integer|min:0',
            'low_stock_threshold' => 'nullable|integer|min:0',
            'category_id' => 'nullable|exists:categories,id',
            'vegetarian' => 'boolean',
            'halal' => 'boolean',
            'gluten_free' => 'boolean',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp,avif|max:4096',
        ]);
        $data['vegetarian'] = (bool)($data['vegetarian'] ?? false);
        $data['halal'] = (bool)($data['halal'] ?? false);
        $data['gluten_free'] = (bool)($data['gluten_free'] ?? false);
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('dishes', 'public');
            $data['image'] = $path; // relative path
        }
        // Avoid DB error if migration hasn't added the column yet
        if (Schema::hasColumn('dishes', 'stock')) {
            $data['stock'] = (int) ($data['stock'] ?? 0);
        } else {
            unset($data['stock']);
        }
        $dish = Dish::create($data);
        return redirect()->route('admin.dishes.index')->with('status','Dish created');
    }

    public function edit(Dish $dish)
    {
        $categories = Category::orderBy('name')->get();
        return view('admin.dishes.edit', compact('dish','categories'));
    }

    public function update(Request $request, Dish $dish)
    {
        $data = $request->validate([
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'stock' => 'nullable|integer|min:0',
            'low_stock_threshold' => 'nullable|integer|min:0',
            'category_id' => 'nullable|exists:categories,id',
            'vegetarian' => 'boolean',
            'halal' => 'boolean',
            'gluten_free' => 'boolean',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp,avif|max:4096',
        ]);
        $data['vegetarian'] = (bool)($data['vegetarian'] ?? false);
        $data['halal'] = (bool)($data['halal'] ?? false);
        $data['gluten_free'] = (bool)($data['gluten_free'] ?? false);
        if ($request->hasFile('image')) {
            if ($dish->image) {
                $oldPath = ltrim($dish->image, '/');
                try { Storage::disk('public')->delete($oldPath); } catch (\Throwable $e) {}
            }
            $path = $request->file('image')->store('dishes', 'public');
            $data['image'] = $path;
        }
        if (Schema::hasColumn('dishes', 'stock')) {
            $data['stock'] = (int) ($data['stock'] ?? (int)($dish->stock ?? 0));
        } else {
            unset($data['stock']);
        }
        $dish->update($data);
        return redirect()->route('admin.dishes.index')->with('status','Dish updated');
    }

    public function destroy(Dish $dish)
    {
        $dish->delete();
        return redirect()->route('admin.dishes.index')->with('status','Dish moved to trash');
    }

    public function trash(Request $request)
    {
        $dishes = Dish::onlyTrashed()->orderByDesc('deleted_at')->paginate(25);
        return view('admin.dishes.trash', compact('dishes'));
    }

    public function restore(int $id)
    {
        $dish = Dish::onlyTrashed()->findOrFail($id);
        $dish->restore();
        return redirect()->route('admin.dishes.trash')->with('status','Dish restored');
    }

    public function forceDelete(int $id)
    {
        $dish = Dish::onlyTrashed()->findOrFail($id);
        // (Optional) delete image file here
        $dish->forceDelete();
        return redirect()->route('admin.dishes.trash')->with('status','Dish permanently deleted');
    }
}
