<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use App\Models\Dish;
use App\Models\Category;

class LandingController extends Controller
{
    /**
     * Display the public landing page with cached, prepared data.
     */
    public function index()
    {
        // 1. Potentially cached dish/fallback slides (legacy behavior)
        $slides = Cache::remember('home:slides:v1', 600, function () {
            try {
                $images = Dish::whereNotNull('image')
                    ->inRandomOrder()
                    ->limit(5)
                    ->pluck('image')
                    ->filter()
                    ->map(fn ($p) => asset('storage/' . $p))
                    ->values()
                    ->all();
            } catch (\Throwable $e) {
                $images = [];
            }
            if (empty($images)) {
                $images = [
                    asset('assets/images/jollof.jpg'),
                    asset('assets/images/egusi.jpg'),
                    asset('assets/images/suya.jpg'),
                    asset('assets/images/efo_riro.jpg'),
                    asset('assets/images/pepper_soup.jpg'),
                ];
            }
            return $images;
        });

        // 2. Override with manifest-driven slider images if present.
        //    Reason slider was previously "broken": Blade only replaced when $slides was empty.
        //    Because controller always provided fallback images, manifest branch never ran.
        try {
            $manifest = \App\Support\ImageManifest::get();
            if ($manifest && !empty($manifest['slider'])) {
                $sliderSlides = collect($manifest['slider'])
                    ->map(function ($entry) {
                        $variants = $entry['variants'] ?? [];
                        if ($variants) {
                            // pick largest width variant
                            usort($variants, fn($a,$b) => ($b['width'] ?? 0) <=> ($a['width'] ?? 0));
                            $largest = $variants[0];
                            if (!empty($largest['jpg'])) {
                                $path = $largest['jpg'];
                                // Provide root-relative path to avoid later double host concatenation
                                if (preg_match('~^https?://~i',$path) || str_starts_with($path,'//')) return $path;
                                return '/'.ltrim($path,'/');
                            }
                        }
                        if (!empty($entry['original'])) {
                            $orig = $entry['original'];
                            if (preg_match('~^https?://~i',$orig) || str_starts_with($orig,'//')) return $orig;
                            return '/'.ltrim($orig,'/');
                        }
                        return null;
                    })
                    ->filter()
                    ->values()
                    ->all();
                if (!empty($sliderSlides)) {
                    $slides = $sliderSlides;
                }
            }
        } catch (\Throwable $e) {
            // Silent failure -> retain previous $slides
        }

        $specials = Cache::remember('home:specials:v1', 600, function () {
            try {
                return Dish::inRandomOrder()
                    ->limit(5)
                    ->get(['id', 'name', 'description', 'price', 'image', 'vegetarian', 'halal']);
            } catch (\Throwable $e) {
                return collect();
            }
        });

        $signature = Cache::remember('home:signature:v1', 600, function () {
            try {
                return Dish::inRandomOrder()
                    ->limit(4)
                    ->get(['id', 'name', 'description', 'price', 'image', 'vegetarian', 'halal']);
            } catch (\Throwable $e) {
                return collect();
            }
        });

        $categories = Cache::remember('home:catnames:v1', 600, function () {
            try {
                return Category::query()->orderBy('name')->pluck('name')->take(6)->all();
            } catch (\Throwable $e) {
                return ['Jollof Rice', 'Suya', 'Egusi Soup', 'Grilled Fish'];
            }
        });

        $menuCats = Cache::remember('home:featuredCats:v1', 600, function () {
            try {
                return Category::with(['dishes' => function ($q) {
                    $q->inRandomOrder()->limit(6);
                }])
                    ->has('dishes')
                    ->inRandomOrder()
                    ->limit(4)
                    ->get();
            } catch (\Throwable $e) {
                return collect();
            }
        });

        $sample = null;
        if ($menuCats->isEmpty()) {
            $sample = [
                ['name' => 'Jollof & Rice', 'items' => [
                    ['n' => 'Classic Jollof Rice', 'd' => 'Smoky, rich tomato base with spices.', 'p' => 3500, 'img' => 'https://lh3.googleusercontent.com/aida-public/AB6AXuAbPnTgfjtqizuZXPdVR3rBGC4gXIBchd2VOHUMiAlcJ2_Mjmx43BcpYw7WQ4jBr2Wpi-CtFyKM9FQldD6K9IcWbAOclYVRnixMvS7vQygEt6bAPCF3ijHnsUlPhLbwHa2moa-lsfuSBK7yXNUbx2BBZQK106iy8t8zAerpqKe6c9zFEkbPSKijwDYotkA9HqtkG6Y9IX6UOaDJLXMkxN_Ip1dm0f4I_SY3xK83ReKcONeLq-ZUhQmiLymRDXD67r8oAr8jdLKIItk'],
                    ['n' => 'Fried Rice Deluxe', 'd' => 'Golden rice with mixed veg and liver.', 'p' => 3800, 'img' => 'https://lh3.googleusercontent.com/aida-public/AB6AXuB49Q0h7NdCmm9BZ3lSYy6EtabICJ6ghW3w0Qd7u4S51_EIl2A2H-sx1HCB_7zXY-9QVQxIpQ6Fpoye039zELAuo0sDfmDOYf62K4om6wM_pxPZof6F291qu4wV1OOeSJabV0njsqN54zDvVA6zSkU7tYpoeZF9gq_1R1GP1-r7I13nFl9wm2yaRYM6AHj_Wm6h_mvfEuuwfTmY857pfGD5hyEvhGo2OxKxZWV2RSmy7V4IecOp02_uMvVuF8O2o0axEET8XDKVmZA'],
                    ['n' => 'Jollof with Chicken', 'd' => 'Served with grilled quarter chicken.', 'p' => 5500, 'img' => 'https://lh3.googleusercontent.com/aida-public/AB6AXuCKtS45Ils9rD0Fz4PH4AElPK2HDfRWusGX6Rh55DhL-Xkh9z8cKVeLj1BnePTWWCL9qWMxXm727ta3-cM_HrGOG5Cs0b6IPOyWj-ZgR4R77DJjZ8Kp6PGeDz3Kws5JzIx9sapo7x4Y0HTEWJurq8ziC0f8W4dYgqP_SEkeBIXkV658V1_kRndLl1LuagwqizR-bAWWB7xDI_cYKPrwCFpwfFWT4szeA_jDOui9NVgUDl6DgYyXDKbIkSC_ktRgvOxCPcDqmVuruXc'],
                ]],
                ['name' => 'Grills & Suya', 'items' => [
                    ['n' => 'Beef Suya Skewers', 'd' => 'Spiced peanut rub, grilled to perfection.', 'p' => 4200, 'img' => 'https://lh3.googleusercontent.com/aida-public/AB6AXuBfJYMVzNCvaAvfzBX2KVWemJFE9ZSTqWDIYTRbYncooy5uc0EEpHLt4O045e3M0UxwR8ywFVBGNHsh88PL0M-t_z7tNhszNVCOMRZmIQ_qCVfS5meTvsy8L7vxOVXwaiAxdpS9ygUoawfgzkYv1nkS6u5SXprvaMTi1taC4doaUqjXtnsccuICYpJphi3CaDvufHMcRBI0cLqRm9U7FH2I0u6kUfZ2ATZt6F0nYLvieXOkCevc2ZYVg7lkror4-nGXDrLhcpOq1yU'],
                    ['n' => 'Chicken Suya Wrap', 'd' => 'Suya-spiced chicken, slaw, soft wrap.', 'p' => 3900, 'img' => 'https://lh3.googleusercontent.com/aida-public/AB6AXuBaH-49KYpOawPWCbNctJEoGRwySgXp4N_qppDWgNe80-199a4-tCpJcTDMqPSWJ5a2Hxr2_K5W73l9kNa21It6WZoPDPIt2sc5DMzdXFYhkef34g2sTgG6YWaJ9j7H_3Q3j6w4F089G7RAeumfKAnr2WO4gzDvWThcE6EAaSMSzH4ywENdOIf-7Tv8yCNbuFLAMmZiHeMYTzROrIQKrfigj5VrH9CUoTYiapTQWbD5Xj4fZO1wF4yEuroC8RFwX_4ZyvvkfqIJxek'],
                    ['n' => 'Mixed Grill Platter', 'd' => 'Beef, chicken, fish, plantains.', 'p' => 8200, 'img' => 'https://lh3.googleusercontent.com/aida-public/AB6AXuBce3tbu0gBuOMFFEnxFYO8ndg2EazUd6XWLvMnZ45wnXTDO27DczVjFeZ3pS_QhtXmxCx0jL_y5pgge8IR2XcCK81IayLy_8u28ZYkHUiHL3Fh_IR9qmWUbQAU9RaUejhuCp9vaQ-AxM0tvHop0HlxFReyNX0NlXsCwdaHC6kmeUKWpPdbXqVQITPaMJ1fbm8GIkK3YeKpbD4SQ9ALR9XY168eQ2Qt6RmP5j9BuiRVXfITZqK0mdRr6i57X8PW-WCVXN9uQkhrMc8'],
                ]],
                ['name' => 'Soups & Swallows', 'items' => [
                    ['n' => 'Egusi with Goat Meat', 'd' => 'Rich melon seed soup; tender goat.', 'p' => 6000, 'img' => '/assets/images/egusi.jpg'],
                    ['n' => 'Efo Riro', 'd' => 'Spinach stew with peppers & proteins.', 'p' => 5200, 'img' => '/assets/images/efo_riro.jpg'],
                    ['n' => 'Pepper Soup', 'd' => 'Aromatic broth, warming spices.', 'p' => 4500, 'img' => '/assets/images/pepper_soup.jpg'],
                ]],
                ['name' => 'Seafood', 'items' => [
                    ['n' => 'Grilled Fish & Plantains', 'd' => 'Smoky fish, sweet plantains.', 'p' => 7000, 'img' => '/assets/images/jollof.jpg'],
                    ['n' => 'Prawn Stir-Fry', 'd' => 'Crunchy prawns, veggies, umami.', 'p' => 7500, 'img' => 'https://images.unsplash.com/photo-1553621042-f6e147245754?w=800&q=60&auto=format'],
                    ['n' => 'Seafood Okra', 'd' => 'Okra stew with mixed seafood.', 'p' => 6800, 'img' => 'https://images.unsplash.com/photo-1512621776951-a57141f2eefd?w=800&q=60&auto=format'],
                ]],
            ];
        }

        return view('landing', compact('slides', 'specials', 'signature', 'categories', 'menuCats', 'sample'));
    }
}
