<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderStatusEvent;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $query = Order::query()->with('user');
        $q = trim((string) $request->get('q', ''));
        $perPage = (int) ($request->get('perPage', 20));
        if (!in_array($perPage, [10,20,25,50,100], true)) { $perPage = 20; }
        $sort = (string) $request->get('sort', 'created');
        $dir = strtolower((string) $request->get('dir', 'desc')) === 'asc' ? 'asc' : 'desc';
        $sortable = [
            'id' => 'orders.id',
            'customer' => 'users.name',
            'payment' => 'orders.payment_method',
            'total' => 'orders.total',
            'status' => 'orders.status',
            'created' => 'orders.created_at',
        ];

        if ($status = $request->string('status')->toString()) {
            $query->where('status', $status);
        }
        if ($pm = $request->string('payment_method')->toString()) {
            $query->where('payment_method', $pm);
        }
        if ($q !== '') {
            $query->leftJoin('users','users.id','=','orders.user_id');
            $query->where(function($w) use ($q){
                $w->where('orders.id', $q)
                  ->orWhere('users.name', 'like', "%$q%")
                  ->orWhere('orders.payment_method', 'like', "%$q%");
            });
            $query->select('orders.*');
        }
        if ($from = $request->date('from')) {
            $query->whereDate('created_at', '>=', $from);
        }
        if ($to = $request->date('to')) {
            $query->whereDate('created_at', '<=', $to);
        }

        $orderBy = $sortable[$sort] ?? $sortable['created'];
        // Ensure join for customer sorting
        if ($orderBy === 'users.name') {
            $query->leftJoin('users','users.id','=','orders.user_id')->select('orders.*');
        }

        $totalMatching = (clone $query)->count();
        $orders = $query->orderBy($orderBy, $dir)->paginate($perPage)->withQueryString();
    $counts = [
            'Received' => Order::where('status','Received')->count(),
            'In Kitchen' => Order::where('status','In Kitchen')->count(),
            'Ready' => Order::where('status','Ready')->count(),
            'Out for Delivery' => Order::where('status','Out for Delivery')->count(),
            'Delivered' => Order::where('status','Delivered')->count(),
        ];
        return view('admin.orders.index', compact('orders','counts','totalMatching','q','perPage','sort','dir'));
    }

    public function show(Order $order)
    {
        $order->load(['user','items.dish','statusEvents.user']);
        return view('admin.orders.show', compact('order'));
    }

    public function print(Order $order)
    {
        $order->load(['user','items.dish']);
        return view('admin.orders.print', compact('order'));
    }

    public function rows(Request $request)
    {
        $query = Order::query()->with('user');
        $q = trim((string) $request->get('q', ''));
        $perPage = (int) ($request->get('perPage', 20));
        if (!in_array($perPage, [10,20,25,50,100], true)) { $perPage = 20; }
        $sort = (string) $request->get('sort', 'created');
        $dir = strtolower((string) $request->get('dir', 'desc')) === 'asc' ? 'asc' : 'desc';
        $sortable = [
            'id' => 'orders.id',
            'customer' => 'users.name',
            'payment' => 'orders.payment_method',
            'total' => 'orders.total',
            'status' => 'orders.status',
            'created' => 'orders.created_at',
        ];
        if ($status = $request->string('status')->toString()) {
            $query->where('status', $status);
        }
        if ($pm = $request->string('payment_method')->toString()) {
            $query->where('payment_method', $pm);
        }
        if ($q !== '') {
            $query->leftJoin('users','users.id','=','orders.user_id');
            $query->where(function($w) use ($q){
                $w->where('orders.id', $q)
                  ->orWhere('users.name', 'like', "%$q%")
                  ->orWhere('orders.payment_method', 'like', "%$q%");
            });
            $query->select('orders.*');
        }
        if ($from = $request->date('from')) {
            $query->whereDate('created_at', '>=', $from);
        }
        if ($to = $request->date('to')) {
            $query->whereDate('created_at', '<=', $to);
        }
        $orderBy = $sortable[$sort] ?? $sortable['created'];
        if ($orderBy === 'users.name') {
            $query->leftJoin('users','users.id','=','orders.user_id')->select('orders.*');
        }
        $orders = $query->orderBy($orderBy, $dir)->paginate($perPage)->withQueryString();
        return view('admin.orders.partials.rows', compact('orders'));
    }

    public function export(Request $request)
    {
        $query = Order::query()->with('user');
        if ($status = $request->string('status')->toString()) {
            $query->where('status', $status);
        }
        if ($pm = $request->string('payment_method')->toString()) {
            $query->where('payment_method', $pm);
        }
        if ($from = $request->date('from')) {
            $query->whereDate('created_at', '>=', $from);
        }
        if ($to = $request->date('to')) {
            $query->whereDate('created_at', '<=', $to);
        }
        $orders = $query->latest()->get();

        $filename = 'orders_export_'.now()->format('Ymd_His').'.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $columns = ['ID','Customer','Payment','Subtotal','Delivery','Discount','Tip','Total','Status','Created At'];

        $callback = function() use ($orders, $columns) {
            $out = fopen('php://output', 'w');
            fputcsv($out, $columns);
            foreach ($orders as $o) {
                fputcsv($out, [
                    $o->id,
                    $o->user?->name ?? 'Guest',
                    $o->payment_method,
                    number_format($o->subtotal, 2, '.', ''),
                    number_format($o->delivery_fee, 2, '.', ''),
                    number_format($o->discount, 2, '.', ''),
                    number_format($o->tip ?? 0, 2, '.', ''),
                    number_format($o->total, 2, '.', ''),
                    $o->status,
                    $o->created_at->format('Y-m-d H:i:s'),
                ]);
            }
            fclose($out);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function update(Request $request, Order $order)
    {
        $data = $request->validate([
            'status' => 'required|in:'.implode(',', Order::STATUSES),
        ]);
        $original = $order->status;
        $order->update($data);
        if ($original !== $order->status) {
            OrderStatusEvent::create([
                'order_id' => $order->id,
                'status' => $order->status,
                'user_id' => $request->user()?->id,
                'note' => $request->string('note')->toString() ?: null,
            ]);
        }
        return back()->with('status','Order updated');
    }

    public function bulkUpdate(Request $request)
    {
        $data = $request->validate([
            'status' => 'required|in:'.implode(',', Order::STATUSES),
            'order_ids' => 'array',
            'order_ids.*' => 'integer|exists:orders,id',
            'select_mode' => 'nullable|in:page,all',
            'filters' => 'nullable|string',
        ]);

        $ordersQuery = Order::query();
        if (($data['select_mode'] ?? 'page') === 'all') {
            // Apply current filters to select all matching
            parse_str($data['filters'] ?? '', $filters);
            if (!empty($filters['status'])) $ordersQuery->where('status', $filters['status']);
            if (!empty($filters['payment_method'])) $ordersQuery->where('payment_method', $filters['payment_method']);
            if (!empty($filters['from'])) $ordersQuery->whereDate('created_at', '>=', $filters['from']);
            if (!empty($filters['to'])) $ordersQuery->whereDate('created_at', '<=', $filters['to']);
        } else {
            $ids = $data['order_ids'] ?? [];
            $ordersQuery->whereIn('id', $ids);
        }

        $orders = $ordersQuery->get();
        foreach ($orders as $order) {
            $original = $order->status;
            if ($original === $data['status']) continue;
            $order->update(['status' => $data['status']]);
            OrderStatusEvent::create([
                'order_id' => $order->id,
                'status' => $order->status,
                'user_id' => $request->user()?->id,
                'note' => $request->string('note')->toString() ?: null,
            ]);
        }

        return back()->with('status', 'Orders updated');
    }
}
