<?php

namespace App\Http\Controllers;

use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class PaystackController extends Controller
{
    public function callback(Request $request)
    {
        $reference = $request->query('reference');
        if (!$reference) {
            return redirect()->route('orders.index')->withErrors(['payment' => 'Missing payment reference.']);
        }

        $secret = config('services.paystack.secret') ?? env('PAYSTACK_SECRET');
        if (!$secret) {
            return redirect()->route('orders.index')->withErrors(['payment' => 'Paystack not configured.']);
        }

        $verify = Http::withToken($secret)
            ->acceptJson()
            ->get('https://api.paystack.co/transaction/verify/'.$reference);

        if (!$verify->ok() || !data_get($verify->json(),'status')) {
            return redirect()->route('orders.index')->withErrors(['payment' => 'Verification failed.']);
        }

        $data = $verify->json('data');
        $orderId = data_get($data, 'metadata.order_id');
        $status = data_get($data, 'status');

        $order = Order::find($orderId);
        if (!$order) {
            return redirect()->route('orders.index')->withErrors(['payment' => 'Order not found.']);
        }

        if ($status === 'success') {
            $order->payment_verified = true;
            $order->status = 'Received';
            $order->save();
            // Clear cart now that payment succeeded
            session()->forget('cart');
            return redirect()->route('orders.show', $order)->with('status','Payment successful. Order received.');
        }

        return redirect()->route('orders.show', $order)->withErrors(['payment' => 'Payment not successful.']);
    }

    public function webhook(Request $request)
    {
        $secret = config('services.paystack.secret') ?? env('PAYSTACK_SECRET');
        if (!$secret) {
            return response('Missing secret', 400);
        }

        $signature = $request->header('X-Paystack-Signature');
        $payload = $request->getContent();
        $computed = hash_hmac('sha512', $payload, $secret);
        if (!$signature || !hash_equals($computed, $signature)) {
            Log::warning('Paystack webhook signature mismatch');
            return response('Invalid signature', 401);
        }

        $data = $request->json('data');
        $event = $request->json('event');
        $reference = data_get($data, 'reference');
        $eventId = data_get($data, 'id');

        // Idempotency: skip if already processed
        $cacheKey = 'paystack_event_'.$eventId;
        if ($eventId && Cache::add($cacheKey, 1, 3600) === false) {
            return response('Already processed', 200);
        }

        if ($event === 'charge.success' && $reference) {
            $orderId = data_get($data, 'metadata.order_id');
            $status = data_get($data, 'status');
            $amount = (int) data_get($data, 'amount', 0);

            $order = Order::find($orderId);
            if ($order) {
                // Basic amount check (kobo)
                $expected = (int) round($order->total * 100);
                if ($amount >= $expected && $status === 'success') {
                    $order->payment_verified = true;
                    $order->payment_reference = $reference;
                    $order->status = 'Received';
                    $order->save();
                }
            }
        }

        return response('ok', 200);
    }
}
