<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Services\QRCodeService;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class QRCodeController extends Controller
{
    /**
     * Generate modern QR code with logo
     */
    public function generate(Request $request)
    {
        $data = $request->get('data', url('/'));
        $size = $request->get('size', 300);
        $format = $request->get('format', 'svg');

        try {
            if ($format === 'svg') {
                return $this->generateSVGQR($data, $size);
            } else {
                return $this->generateImageQR($data, $size);
            }
        } catch (\Exception $e) {
            return response()->json(['error' => 'QR code generation failed'], 500);
        }
    }

    /**
     * Generate modern SVG QR code
     */
    private function generateSVGQR(string $data, int $size): Response
    {
        $logoUrl = asset('assets/logo.png');
        $qrApiUrl = "https://api.qrserver.com/v1/create-qr-code/?size=" . ($size-60) . "x" . ($size-60) . "&data=" . urlencode($data) . "&format=svg&bgcolor=transparent&color=1f2937&qzone=1&margin=0";
        
        // Pre-calculate positions to avoid PHP syntax errors in SVG
        $centerX = $size / 2;
        $centerY = $size / 2;
        $qrInnerSize = $size - 60;
        $logoX = $centerX - 30;
        $logoY = $centerY - 30;
        $cornerRight = $size - 28;
        $cornerBottom = $size - 28;
        
        $svg = <<<SVG
<svg width="{$size}" height="{$size}" viewBox="0 0 {$size} {$size}" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
  <defs>
    <style>
      .qr-container { 
        filter: drop-shadow(0 4px 6px rgba(0, 0, 0, 0.1)); 
      }
      .logo-bg { 
        fill: white; 
        filter: drop-shadow(0 2px 8px rgba(0, 0, 0, 0.15)); 
      }
      .logo-border {
        fill: none;
        stroke: #e5e7eb;
        stroke-width: 2;
      }
      .qr-bg {
        fill: url(#qrGradient);
      }
    </style>
    
    <!-- Gradient definitions -->
    <linearGradient id="qrGradient" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:#ffffff;stop-opacity:1" />
      <stop offset="100%" style="stop-color:#f8fafc;stop-opacity:1" />
    </linearGradient>
    
    <!-- Rounded corners clip path -->
    <clipPath id="roundedCorners">
      <rect width="{$size}" height="{$size}" rx="24" ry="24"/>
    </clipPath>
    
    <!-- QR code clip path (to avoid logo area) -->
    <clipPath id="qrClip">
      <rect width="{$size}" height="{$size}" rx="24" ry="24"/>
      <circle cx="{$centerX}" cy="{$centerY}" r="45" fill="black"/>
    </clipPath>
  </defs>
  
  <!-- Background -->
  <rect width="{$size}" height="{$size}" rx="24" ry="24" class="qr-bg"/>
  
  <!-- QR Code pattern -->
  <g class="qr-container" clip-path="url(#roundedCorners)">
    <foreignObject x="30" y="30" width="{$qrInnerSize}" height="{$qrInnerSize}">
      <div xmlns="http://www.w3.org/1999/xhtml" style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center;">
        <img src="{$qrApiUrl}" width="{$qrInnerSize}" height="{$qrInnerSize}" style="max-width: 100%; max-height: 100%; object-fit: contain;"/>
      </div>
    </foreignObject>
  </g>
  
  <!-- Logo background circle -->
  <circle cx="{$centerX}" cy="{$centerY}" r="42" class="logo-bg"/>
  <circle cx="{$centerX}" cy="{$centerY}" r="42" class="logo-border"/>
  
  <!-- Logo -->
  <foreignObject x="{$logoX}" y="{$logoY}" width="60" height="60">
    <div xmlns="http://www.w3.org/1999/xhtml" style="width: 60px; height: 60px; display: flex; align-items: center; justify-content: center; border-radius: 12px; overflow: hidden;">
      <img src="{$logoUrl}" width="50" height="50" style="object-fit: contain; border-radius: 8px;"/>
    </div>
  </foreignObject>
  
  <!-- Modern corner elements for visual appeal -->
  <g opacity="0.1">
    <rect x="20" y="20" width="8" height="8" rx="2" fill="#6366f1"/>
    <rect x="{$cornerRight}" y="20" width="8" height="8" rx="2" fill="#06b6d4"/>
    <rect x="20" y="{$cornerBottom}" width="8" height="8" rx="2" fill="#10b981"/>
    <rect x="{$cornerRight}" y="{$cornerBottom}" width="8" height="8" rx="2" fill="#f59e0b"/>
  </g>
</svg>
SVG;

        return response($svg)
            ->header('Content-Type', 'image/svg+xml')
            ->header('Cache-Control', 'public, max-age=3600');
    }

    /**
     * Generate image QR code with logo overlay
     */
    private function generateImageQR(string $data, int $size): Response
    {
        // For image format, redirect to enhanced API
        $qrUrl = "https://api.qrserver.com/v1/create-qr-code/?size={$size}x{$size}&data=" . urlencode($data) . "&format=png&bgcolor=FFFFFF&color=1f2937&qzone=2&margin=10";
        
        return redirect($qrUrl);
    }

    /**
     * Generate PWA install QR code
     */
    public function pwaInstall(Request $request)
    {
        $size = $request->get('size', 320);
        $appUrl = url('/');
        
        return $this->generateSVGQR($appUrl, $size);
    }

    /**
     * Generate contact QR code
     */
    public function contact(Request $request)
    {
        $size = $request->get('size', 280);
        
        // Create vCard data for contact
        $vcard = "BEGIN:VCARD\n";
        $vcard .= "VERSION:3.0\n";
        $vcard .= "FN:Bonny Delight Restaurant\n";
        $vcard .= "ORG:Bonny Delight\n";
        $vcard .= "TEL:+2348000000000\n";
        $vcard .= "URL:" . url('/') . "\n";
        $vcard .= "EMAIL:info@bonnydelight.com\n";
        $vcard .= "ADR:;;123 Restaurant Street;Lagos;Lagos;100001;Nigeria\n";
        $vcard .= "END:VCARD";
        
        return $this->generateSVGQR($vcard, $size);
    }
}
