<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use App\Models\Dish;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AnalyticsController extends Controller
{
    public function index()
    {
        return view('super-admin.analytics.index');
    }
    
    public function revenue(Request $request)
    {
        $period = $request->get('period', '30d');
        $startDate = $this->getStartDate($period);
        
        // Revenue over time with comparison to previous period
        $revenueData = Order::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('SUM(total) as revenue'),
                DB::raw('COUNT(*) as orders'),
                DB::raw('AVG(total) as avg_order_value')
            )
            ->where('status', 'completed')
            ->where('created_at', '>=', $startDate)
            ->groupBy('date')
            ->orderBy('date')
            ->get();
        
        // Previous period comparison
        $previousStartDate = $this->getPreviousPeriodStart($period, $startDate);
        $previousRevenue = Order::where('status', 'completed')
            ->whereBetween('created_at', [$previousStartDate, $startDate])
            ->sum('total');
        
        $currentRevenue = Order::where('status', 'completed')
            ->where('created_at', '>=', $startDate)
            ->sum('total');
        
        $revenueGrowth = $previousRevenue > 0 ? (($currentRevenue - $previousRevenue) / $previousRevenue) * 100 : 0;
        
        // Revenue by category with growth
        $categoryRevenue = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('categories', 'dishes.category_id', '=', 'categories.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('orders.status', 'completed')
            ->where('orders.created_at', '>=', $startDate)
            ->select(
                'categories.id',
                'categories.name',
                DB::raw('SUM(order_items.quantity * order_items.price) as revenue'),
                DB::raw('SUM(order_items.quantity) as items_sold'),
                DB::raw('COUNT(DISTINCT orders.id) as orders_count'),
                DB::raw('AVG(order_items.price) as avg_price')
            )
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('revenue')
            ->get();
        
        // Hour-by-hour revenue analysis
        $hourlyRevenue = Order::select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('SUM(total) as revenue'),
                DB::raw('COUNT(*) as orders')
            )
            ->where('status', 'completed')
            ->where('created_at', '>=', $startDate)
            ->groupBy('hour')
            ->orderBy('hour')
            ->get();
        
        // Day of week analysis
        $weekdayRevenue = Order::select(
                DB::raw('DAYOFWEEK(created_at) as day_of_week'),
                DB::raw('DAYNAME(created_at) as day_name'),
                DB::raw('SUM(total) as revenue'),
                DB::raw('COUNT(*) as orders'),
                DB::raw('AVG(total) as avg_order_value')
            )
            ->where('status', 'completed')
            ->where('created_at', '>=', $startDate)
            ->groupBy('day_of_week', 'day_name')
            ->orderBy('day_of_week')
            ->get();
        
        return response()->json([
            'revenue_data' => $revenueData,
            'category_revenue' => $categoryRevenue,
            'hourly_revenue' => $hourlyRevenue,
            'weekday_revenue' => $weekdayRevenue,
            'revenue_growth' => round($revenueGrowth, 2),
            'current_revenue' => $currentRevenue,
            'previous_revenue' => $previousRevenue
        ]);
    }
    
    public function customers(Request $request)
    {
        $period = $request->get('period', '30d');
        $startDate = $this->getStartDate($period);
        
        // Customer acquisition with cohort analysis
        $newCustomers = User::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as new_customers')
            )
            ->where('created_at', '>=', $startDate)
            ->groupBy('date')
            ->orderBy('date')
            ->get();
        
        // Customer lifetime value and segmentation
        $customerLTV = DB::table('users')
            ->join('orders', 'users.id', '=', 'orders.user_id')
            ->where('orders.status', 'completed')
            ->where('orders.created_at', '>=', $startDate)
            ->select(
                'users.id',
                'users.name',
                'users.email',
                'users.created_at as join_date',
                DB::raw('COUNT(orders.id) as total_orders'),
                DB::raw('SUM(orders.total) as lifetime_value'),
                DB::raw('AVG(orders.total) as avg_order_value'),
                DB::raw('MAX(orders.created_at) as last_order_date'),
                DB::raw('DATEDIFF(MAX(orders.created_at), MIN(orders.created_at)) as customer_lifespan_days')
            )
            ->groupBy('users.id', 'users.name', 'users.email', 'users.created_at')
            ->orderByDesc('lifetime_value')
            ->take(20)
            ->get();
        
        // Customer segments
        $customerSegments = DB::table('users')
            ->join('orders', 'users.id', '=', 'orders.user_id')
            ->where('orders.status', 'completed')
            ->select(
                DB::raw('CASE 
                    WHEN COUNT(orders.id) = 1 THEN "One-time"
                    WHEN COUNT(orders.id) BETWEEN 2 AND 5 THEN "Regular"
                    WHEN COUNT(orders.id) BETWEEN 6 AND 10 THEN "Loyal"
                    ELSE "VIP"
                END as segment'),
                DB::raw('COUNT(DISTINCT users.id) as customer_count'),
                DB::raw('SUM(orders.total) as total_revenue'),
                DB::raw('AVG(orders.total) as avg_order_value')
            )
            ->groupBy(DB::raw('CASE 
                WHEN COUNT(orders.id) = 1 THEN "One-time"
                WHEN COUNT(orders.id) BETWEEN 2 AND 5 THEN "Regular"
                WHEN COUNT(orders.id) BETWEEN 6 AND 10 THEN "Loyal"
                ELSE "VIP"
            END'))
            ->havingRaw('COUNT(orders.id) >= 1')
            ->get();
        
        // Customer retention analysis
        $totalCustomers = User::whereHas('orders')->count();
        $repeatCustomers = User::whereHas('orders', function($query) {
            $query->havingRaw('COUNT(*) > 1');
        })->count();
        $repeatRate = $totalCustomers > 0 ? ($repeatCustomers / $totalCustomers) * 100 : 0;
        
        // Customer churn analysis (customers who haven't ordered in 30+ days)
        $churnThreshold = Carbon::now()->subDays(30);
        $churnedCustomers = User::whereHas('orders', function($query) use ($churnThreshold) {
            $query->where('created_at', '<', $churnThreshold);
        })->whereDoesntHave('orders', function($query) use ($churnThreshold) {
            $query->where('created_at', '>=', $churnThreshold);
        })->count();
        
        $churnRate = $totalCustomers > 0 ? ($churnedCustomers / $totalCustomers) * 100 : 0;
        
        return response()->json([
            'new_customers' => $newCustomers,
            'customer_ltv' => $customerLTV,
            'customer_segments' => $customerSegments,
            'repeat_rate' => round($repeatRate, 2),
            'churn_rate' => round($churnRate, 2),
            'total_customers' => $totalCustomers,
            'churned_customers' => $churnedCustomers
        ]);
    }
    
    public function products(Request $request)
    {
        $period = $request->get('period', '30d');
        $startDate = $this->getStartDate($period);
        
        // Best selling products with profitability
        $bestSellers = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('categories', 'dishes.category_id', '=', 'categories.id')
            ->where('orders.status', 'completed')
            ->where('orders.created_at', '>=', $startDate)
            ->select(
                'dishes.id',
                'dishes.name',
                'dishes.price',
                'categories.name as category',
                DB::raw('SUM(order_items.quantity) as total_sold'),
                DB::raw('SUM(order_items.quantity * order_items.price) as revenue'),
                DB::raw('AVG(order_items.price) as avg_selling_price'),
                DB::raw('COUNT(DISTINCT orders.id) as orders_count'),
                DB::raw('SUM(order_items.quantity) / COUNT(DISTINCT orders.id) as avg_quantity_per_order')
            )
            ->groupBy('dishes.id', 'dishes.name', 'dishes.price', 'categories.name')
            ->orderByDesc('total_sold')
            ->take(20)
            ->get();
        
        // Product performance matrix (Revenue vs Frequency)
        $productMatrix = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('orders.status', 'completed')
            ->where('orders.created_at', '>=', $startDate)
            ->select(
                'dishes.name',
                DB::raw('SUM(order_items.quantity * order_items.price) as revenue'),
                DB::raw('COUNT(DISTINCT orders.id) as frequency'),
                DB::raw('SUM(order_items.quantity) as total_quantity')
            )
            ->groupBy('dishes.id', 'dishes.name')
            ->orderByDesc('revenue')
            ->get();
        
        // Inventory turnover analysis
        $inventoryTurnover = DB::table('dishes')
            ->leftJoin('order_items', function($join) use ($startDate) {
                $join->on('dishes.id', '=', 'order_items.dish_id')
                     ->join('orders', 'order_items.order_id', '=', 'orders.id')
                     ->where('orders.status', 'completed')
                     ->where('orders.created_at', '>=', $startDate);
            })
            ->select(
                'dishes.id',
                'dishes.name',
                'dishes.stock',
                'dishes.price',
                DB::raw('COALESCE(SUM(order_items.quantity), 0) as sold'),
                DB::raw('CASE 
                    WHEN dishes.stock > 0 THEN COALESCE(SUM(order_items.quantity), 0) / dishes.stock 
                    ELSE 0 
                END as turnover_ratio'),
                DB::raw('dishes.stock * dishes.price as inventory_value')
            )
            ->groupBy('dishes.id', 'dishes.name', 'dishes.stock', 'dishes.price')
            ->orderByDesc(DB::raw('COALESCE(SUM(order_items.quantity), 0) / dishes.stock'))
            ->get();
        
        // Category performance comparison
        $categoryPerformance = DB::table('categories')
            ->leftJoin('dishes', 'categories.id', '=', 'dishes.category_id')
            ->leftJoin('order_items', function($join) use ($startDate) {
                $join->on('dishes.id', '=', 'order_items.dish_id')
                     ->join('orders', 'order_items.order_id', '=', 'orders.id')
                     ->where('orders.status', 'completed')
                     ->where('orders.created_at', '>=', $startDate);
            })
            ->select(
                'categories.name',
                DB::raw('COUNT(DISTINCT dishes.id) as product_count'),
                DB::raw('COALESCE(SUM(order_items.quantity), 0) as items_sold'),
                DB::raw('COALESCE(SUM(order_items.quantity * order_items.price), 0) as revenue'),
                DB::raw('COUNT(DISTINCT order_items.order_id) as orders_count')
            )
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('revenue')
            ->get();
        
        return response()->json([
            'best_sellers' => $bestSellers,
            'product_matrix' => $productMatrix,
            'inventory_turnover' => $inventoryTurnover,
            'category_performance' => $categoryPerformance
        ]);
    }
    
    public function operational(Request $request)
    {
        $period = $request->get('period', '30d');
        $startDate = $this->getStartDate($period);
        
        // Order fulfillment metrics
        $fulfillmentMetrics = Order::select(
                'status',
                DB::raw('COUNT(*) as count'),
                DB::raw('AVG(TIMESTAMPDIFF(MINUTE, created_at, updated_at)) as avg_processing_time'),
                DB::raw('SUM(total) as revenue')
            )
            ->where('created_at', '>=', $startDate)
            ->groupBy('status')
            ->get();
        
        // Peak hours analysis
        $peakHours = Order::select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('COUNT(*) as order_count'),
                DB::raw('SUM(total) as revenue'),
                DB::raw('AVG(total) as avg_order_value')
            )
            ->where('created_at', '>=', $startDate)
            ->groupBy('hour')
            ->orderBy('hour')
            ->get();
        
        // Order size distribution
        $orderSizeDistribution = Order::select(
                DB::raw('CASE 
                    WHEN total < 1000 THEN "Small (< ₦1,000)"
                    WHEN total BETWEEN 1000 AND 3000 THEN "Medium (₦1,000-3,000)"
                    WHEN total BETWEEN 3000 AND 5000 THEN "Large (₦3,000-5,000)"
                    ELSE "XL (> ₦5,000)"
                END as size_category'),
                DB::raw('COUNT(*) as order_count'),
                DB::raw('SUM(total) as revenue'),
                DB::raw('AVG(total) as avg_value')
            )
            ->where('status', 'completed')
            ->where('created_at', '>=', $startDate)
            ->groupBy(DB::raw('CASE 
                WHEN total < 1000 THEN "Small (< ₦1,000)"
                WHEN total BETWEEN 1000 AND 3000 THEN "Medium (₦1,000-3,000)"
                WHEN total BETWEEN 3000 AND 5000 THEN "Large (₦3,000-5,000)"
                ELSE "XL (> ₦5,000)"
            END'))
            ->get();
        
        return response()->json([
            'fulfillment_metrics' => $fulfillmentMetrics,
            'peak_hours' => $peakHours,
            'order_size_distribution' => $orderSizeDistribution
        ]);
    }
    
    public function financial(Request $request)
    {
        $period = $request->get('period', '30d');
        $startDate = $this->getStartDate($period);
        
        // Revenue breakdown
        $revenueBreakdown = Order::select(
                DB::raw('SUM(subtotal) as gross_revenue'),
                DB::raw('SUM(delivery_fee) as delivery_revenue'),
                DB::raw('SUM(service_charge) as service_revenue'),
                DB::raw('SUM(discount) as total_discounts'),
                DB::raw('SUM(total) as net_revenue'),
                DB::raw('COUNT(*) as total_orders')
            )
            ->where('status', 'completed')
            ->where('created_at', '>=', $startDate)
            ->first();
        
        // Cost analysis (estimated based on product costs)
        $costAnalysis = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('orders.status', 'completed')
            ->where('orders.created_at', '>=', $startDate)
            ->select(
                DB::raw('SUM(order_items.quantity * order_items.price) as revenue'),
                DB::raw('SUM(order_items.quantity * (dishes.price * 0.4)) as estimated_costs'),
                DB::raw('SUM(order_items.quantity * (dishes.price * 0.6)) as estimated_profit')
            )
            ->first();
        
        // Profit margins by category
        $categoryProfits = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('categories', 'dishes.category_id', '=', 'categories.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('orders.status', 'completed')
            ->where('orders.created_at', '>=', $startDate)
            ->select(
                'categories.name',
                DB::raw('SUM(order_items.quantity * order_items.price) as revenue'),
                DB::raw('SUM(order_items.quantity * (dishes.price * 0.4)) as estimated_costs'),
                DB::raw('(SUM(order_items.quantity * order_items.price) - SUM(order_items.quantity * (dishes.price * 0.4))) as profit'),
                DB::raw('((SUM(order_items.quantity * order_items.price) - SUM(order_items.quantity * (dishes.price * 0.4))) / SUM(order_items.quantity * order_items.price)) * 100 as profit_margin')
            )
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('profit')
            ->get();
        
        return response()->json([
            'revenue_breakdown' => $revenueBreakdown,
            'cost_analysis' => $costAnalysis,
            'category_profits' => $categoryProfits
        ]);
    }
    
    public function realtime()
    {
        // Real-time metrics for dashboard updates
        $today = Carbon::today();
        
        $todayMetrics = [
            'orders_today' => Order::whereDate('created_at', $today)->count(),
            'revenue_today' => Order::where('status', 'completed')->whereDate('created_at', $today)->sum('total'),
            'new_customers_today' => User::whereDate('created_at', $today)->count(),
            'pending_orders' => Order::where('status', 'pending')->count(),
            'avg_order_value_today' => Order::where('status', 'completed')->whereDate('created_at', $today)->avg('total')
        ];
        
        // Recent activity
        $recentOrders = Order::with(['user:id,name', 'items.dish:id,name'])
            ->latest()
            ->take(5)
            ->get();
        
        return response()->json([
            'today_metrics' => $todayMetrics,
            'recent_orders' => $recentOrders
        ]);
    }
    
    private function getStartDate($period)
    {
        return match($period) {
            '7d' => Carbon::now()->subDays(7),
            '30d' => Carbon::now()->subDays(30),
            '90d' => Carbon::now()->subDays(90),
            '1y' => Carbon::now()->subYear(),
            default => Carbon::now()->subDays(30)
        };
    }
    
    private function getPreviousPeriodStart($period, $startDate)
    {
        $daysDiff = Carbon::now()->diffInDays($startDate);
        return Carbon::parse($startDate)->subDays($daysDiff);
    }
}