<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use App\Models\Dish;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
// use Barryvdh\DomPDF\Facade\Pdf; // Temporarily disabled until package is installed

class ReportsController extends Controller
{
    public function index()
    {
        return view('super-admin.reports.index');
    }
    
    public function salesReport(Request $request)
    {
        // Provide defaults if parameters are missing
        $request->merge([
            'start_date' => $request->get('start_date', now()->subDays(30)->format('Y-m-d')),
            'end_date' => $request->get('end_date', now()->format('Y-m-d')),
            'format' => $request->get('format', 'view')
        ]);
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'format' => 'in:view,pdf,csv'
        ]);
        
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        // Get orders with detailed relationships
        $orders = Order::with(['user', 'items.dish.category'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Calculate comprehensive sales metrics
        $completedOrders = $orders->where('status', 'completed');
        $cancelledOrders = $orders->where('status', 'cancelled');
        $pendingOrders = $orders->whereIn('status', ['pending', 'confirmed', 'preparing']);
        
        // Daily sales breakdown
        $dailySales = $completedOrders->groupBy(function($order) {
            return $order->created_at->format('Y-m-d');
        })->map(function($dayOrders) {
            return [
                'orders' => $dayOrders->count(),
                'revenue' => $dayOrders->sum('total'),
                'items' => $dayOrders->sum(function($order) {
                    return optional($order->items)->sum('quantity') ?? 0;
                })
            ];
        })->sortKeys();
        
        // Top selling dishes
        $topDishes = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('categories', 'dishes.category_id', '=', 'categories.id')
            ->where('orders.status', 'completed')
            ->whereBetween('orders.created_at', [$startDate, $endDate])
            ->select(
                'dishes.name',
                'categories.name as category',
                'dishes.price',
                DB::raw('SUM(order_items.quantity) as total_sold'),
                DB::raw('SUM(order_items.quantity * order_items.price) as total_revenue')
            )
            ->groupBy('dishes.id', 'dishes.name', 'categories.name', 'dishes.price')
            ->orderByDesc('total_sold')
            ->limit(10)
            ->get();
        
        // Revenue by category
        $categoryRevenue = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('categories', 'dishes.category_id', '=', 'categories.id')
            ->where('orders.status', 'completed')
            ->whereBetween('orders.created_at', [$startDate, $endDate])
            ->select(
                'categories.name',
                DB::raw('SUM(order_items.quantity * order_items.price) as revenue'),
                DB::raw('SUM(order_items.quantity) as items_sold'),
                DB::raw('COUNT(DISTINCT orders.id) as orders_count')
            )
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('revenue')
            ->get();
        
        // Hour-based analysis
        $hourlyDistribution = $completedOrders->groupBy(function($order) {
            return $order->created_at->format('H');
        })->map(function($hourOrders, $hour) {
            return [
                'hour' => $hour . ':00',
                'orders' => $hourOrders->count(),
                'revenue' => $hourOrders->sum('total')
            ];
        })->sortKeys();
        
        $summary = [
            'total_orders' => $orders->count(),
            'completed_orders' => $completedOrders->count(),
            'pending_orders' => $pendingOrders->count(),
            'cancelled_orders' => $cancelledOrders->count(),
            'total_revenue' => $completedOrders->sum('total'),
            'avg_order_value' => $completedOrders->avg('total') ?: 0,
            'total_items_sold' => $completedOrders->sum(function($order) {
                return optional($order->items)->sum('quantity') ?? 0;
            }),
            'conversion_rate' => $orders->count() > 0 ? ($completedOrders->count() / $orders->count()) * 100 : 0,
            'period' => $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y'),
            'days_in_period' => $startDate->diffInDays($endDate) + 1
        ];
        
        $data = compact('orders', 'summary', 'dailySales', 'topDishes', 'categoryRevenue', 'hourlyDistribution');
        
        if ($request->format === 'pdf') {
            // PDF generation temporarily disabled - DomPDF package needs to be installed
            return response()->json([
                'error' => 'PDF generation is currently unavailable. Please contact administrator to install the required packages.',
                'message' => 'Use CSV export as an alternative.'
            ], 503);
        }
        
        if ($request->format === 'csv') {
            return $this->exportSalesCSV($orders, $summary);
        }
        
        return view('super-admin.reports.sales', $data);
    }
    
    public function customerReport(Request $request)
    {
        // Provide defaults if parameters are missing
        $request->merge([
            'start_date' => $request->get('start_date', now()->subDays(30)->format('Y-m-d')),
            'end_date' => $request->get('end_date', now()->format('Y-m-d')),
            'format' => $request->get('format', 'view')
        ]);
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'format' => 'in:view,pdf,csv'
        ]);
        
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        // Customer analytics with detailed metrics
        $customers = User::with(['orders' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate])
                      ->where('status', 'completed');
            }])
            ->whereHas('orders', function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->get()
            ->map(function($customer) use ($startDate, $endDate) {
                $orders = $customer->orders;
                $allOrders = $customer->orders()->where('status', 'completed')->get();
                
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'phone' => $customer->phone ?? 'N/A',
                    'registration_date' => $customer->created_at,
                    'total_orders' => $orders->count(),
                    'total_spent' => $orders->sum('total'),
                    'avg_order_value' => $orders->avg('total') ?: 0,
                    'first_order' => $orders->min('created_at'),
                    'last_order' => $orders->max('created_at'),
                    'lifetime_orders' => $allOrders->count(),
                    'lifetime_value' => $allOrders->sum('total'),
                    'days_since_last_order' => $orders->isNotEmpty() ? 
                        Carbon::parse($orders->max('created_at'))->diffInDays(now()) : null,
                    'frequency_score' => $orders->count() / max(1, $startDate->diffInDays($endDate) + 1) * 30 // Orders per month
                ];
            })
            ->sortByDesc('total_spent');
        
        // New customers in period
        $newCustomers = User::whereBetween('created_at', [$startDate, $endDate])
            ->with(['orders' => function($query) {
                $query->where('status', 'completed');
            }])
            ->get()
            ->map(function($customer) {
                $orders = $customer->orders;
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'registration_date' => $customer->created_at,
                    'first_order_date' => $orders->min('created_at'),
                    'orders_count' => $orders->count(),
                    'total_spent' => $orders->sum('total'),
                    'days_to_first_order' => $orders->isNotEmpty() ? 
                        $customer->created_at->diffInDays(Carbon::parse($orders->min('created_at'))) : null
                ];
            });
        
        // Customer segmentation
        $segmentation = [
            'new_customers' => $customers->filter(function($c) { return $c['lifetime_orders'] <= 1; })->count(),
            'returning_customers' => $customers->filter(function($c) { return $c['lifetime_orders'] > 1 && $c['lifetime_orders'] <= 5; })->count(),
            'loyal_customers' => $customers->filter(function($c) { return $c['lifetime_orders'] > 5; })->count(),
            'high_value' => $customers->filter(function($c) { return $c['total_spent'] > 1000; })->count(),
            'at_risk' => $customers->filter(function($c) { return $c['days_since_last_order'] > 30; })->count()
        ];
        
        // Order frequency analysis
        $frequencyAnalysis = $customers->groupBy(function($customer) {
            if ($customer['total_orders'] == 1) return '1 order';
            if ($customer['total_orders'] <= 3) return '2-3 orders';
            if ($customer['total_orders'] <= 5) return '4-5 orders';
            if ($customer['total_orders'] <= 10) return '6-10 orders';
            return '10+ orders';
        })->map(function($group) {
            return [
                'count' => $group->count(),
                'total_revenue' => $group->sum('total_spent'),
                'avg_value' => $group->avg('total_spent')
            ];
        });
        
        $summary = [
            'total_customers' => $customers->count(),
            'new_customers' => $newCustomers->count(),
            'total_revenue' => $customers->sum('total_spent'),
            'avg_customer_value' => $customers->avg('total_spent') ?: 0,
            'avg_order_frequency' => $customers->avg('total_orders') ?: 0,
            'avg_order_value' => $customers->avg('avg_order_value') ?: 0,
            'customer_retention_rate' => $customers->count() > 0 ? 
                ($customers->filter(function($c) { return $c['total_orders'] > 1; })->count() / $customers->count()) * 100 : 0,
            'period' => $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y')
        ];
        
        $data = compact('customers', 'newCustomers', 'summary', 'segmentation', 'frequencyAnalysis');
        
        if ($request->format === 'pdf') {
            // PDF generation temporarily disabled - DomPDF package needs to be installed
            return response()->json([
                'error' => 'PDF generation is currently unavailable. Please contact administrator to install the required packages.',
                'message' => 'Use CSV export as an alternative.'
            ], 503);
        }
        
        if ($request->format === 'csv') {
            return $this->exportCustomersCSV($customers, $summary);
        }
        
        return view('super-admin.reports.customers', $data);
    }
    
    public function productReport(Request $request)
    {
        // Provide defaults if parameters are missing
        $request->merge([
            'start_date' => $request->get('start_date', now()->subDays(30)->format('Y-m-d')),
            'end_date' => $request->get('end_date', now()->format('Y-m-d')),
            'format' => $request->get('format', 'view')
        ]);
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'format' => 'in:view,pdf,csv'
        ]);
        
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        // Product performance with comprehensive metrics
        $products = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('categories', 'dishes.category_id', '=', 'categories.id')
            ->where('orders.status', 'completed')
            ->whereBetween('orders.created_at', [$startDate, $endDate])
            ->select(
                'dishes.id',
                'dishes.name',
                'categories.name as category',
                'dishes.price',
                'dishes.stock',
                DB::raw('SUM(order_items.quantity) as total_sold'),
                DB::raw('SUM(order_items.quantity * order_items.price) as total_revenue'),
                DB::raw('AVG(order_items.price) as avg_price'),
                DB::raw('COUNT(DISTINCT orders.id) as orders_count'),
                DB::raw('MIN(orders.created_at) as first_sale'),
                DB::raw('MAX(orders.created_at) as last_sale')
            )
            ->groupBy('dishes.id', 'dishes.name', 'categories.name', 'dishes.price', 'dishes.stock')
            ->orderByDesc('total_revenue')
            ->get();
        
        // Category performance
        $categoryPerformance = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('categories', 'dishes.category_id', '=', 'categories.id')
            ->where('orders.status', 'completed')
            ->whereBetween('orders.created_at', [$startDate, $endDate])
            ->select(
                'categories.id',
                'categories.name',
                DB::raw('SUM(order_items.quantity * order_items.price) as revenue'),
                DB::raw('SUM(order_items.quantity) as items_sold'),
                DB::raw('COUNT(DISTINCT dishes.id) as unique_dishes'),
                DB::raw('COUNT(DISTINCT orders.id) as orders_count'),
                DB::raw('AVG(order_items.price) as avg_price')
            )
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('revenue')
            ->get();
        
        // Low stock analysis
        $lowStockDishes = Dish::with('category')
            ->where('stock', '<=', 10)
            ->select('id', 'name', 'category_id', 'stock', 'price')
            ->orderBy('stock', 'asc')
            ->get();
        
        // Get sales velocity for stock analysis
        foreach ($lowStockDishes as $dish) {
            $soldInPeriod = DB::table('order_items')
                ->join('orders', 'order_items.order_id', '=', 'orders.id')
                ->where('order_items.dish_id', $dish->id)
                ->where('orders.status', 'completed')
                ->whereBetween('orders.created_at', [$startDate, $endDate])
                ->sum('order_items.quantity');
                
            $dish->sold_in_period = $soldInPeriod;
            $dish->days_in_period = $startDate->diffInDays($endDate) + 1;
            $dish->daily_velocity = $dish->days_in_period > 0 ? $soldInPeriod / $dish->days_in_period : 0;
            $dish->estimated_days_until_stockout = $dish->daily_velocity > 0 ? $dish->stock / $dish->daily_velocity : null;
        }
        
        // Performance trends
        $performanceTrends = $products->map(function($product) use ($startDate, $endDate) {
            $daysInPeriod = $startDate->diffInDays($endDate) + 1;
            $dailyAverage = $product->total_sold / $daysInPeriod;
            
            // Calculate performance score (revenue per day normalized)
            $revenuePerDay = $product->total_revenue / $daysInPeriod;
            
            return [
                'name' => $product->name,
                'category' => $product->category,
                'daily_sales' => round($dailyAverage, 2),
                'daily_revenue' => round($revenuePerDay, 2),
                'revenue_per_item' => $product->total_sold > 0 ? round($product->total_revenue / $product->total_sold, 2) : 0,
                'market_share' => 0 // Will be calculated below
            ];
        });
        
        // Calculate market share
        $totalRevenue = $products->sum('total_revenue');
        $performanceTrends = $performanceTrends->map(function($trend) use ($totalRevenue, $products) {
            $productRevenue = $products->where('name', $trend['name'])->first()->total_revenue ?? 0;
            $trend['market_share'] = $totalRevenue > 0 ? round(($productRevenue / $totalRevenue) * 100, 2) : 0;
            return $trend;
        });
        
        $summary = [
            'total_products' => $products->count(),
            'total_items_sold' => $products->sum('total_sold'),
            'total_revenue' => $products->sum('total_revenue'),
            'avg_revenue_per_product' => $products->avg('total_revenue') ?: 0,
            'avg_items_per_product' => $products->avg('total_sold') ?: 0,
            'low_stock_items' => $lowStockDishes->count(),
            'critical_stock_items' => $lowStockDishes->where('stock', '<=', 5)->count(),
            'top_category' => $categoryPerformance->first()->name ?? 'N/A',
            'period' => $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y')
        ];
        
        $data = compact('products', 'categoryPerformance', 'lowStockDishes', 'performanceTrends', 'summary');
        
        if ($request->format === 'pdf') {
            // PDF generation temporarily disabled - DomPDF package needs to be installed
            return response()->json([
                'error' => 'PDF generation is currently unavailable. Please contact administrator to install the required packages.',
                'message' => 'Use CSV export as an alternative.'
            ], 503);
        }
        
        if ($request->format === 'csv') {
            return $this->exportProductsCSV($products, $summary);
        }
        
        return view('super-admin.reports.products', $data);
    }
    
    public function inventoryReport(Request $request)
    {
        // Provide defaults if parameters are missing
        $request->merge([
            'start_date' => $request->get('start_date', now()->subDays(30)->format('Y-m-d')),
            'end_date' => $request->get('end_date', now()->format('Y-m-d')),
            'format' => $request->get('format', 'view')
        ]);
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'format' => 'in:view,pdf,csv'
        ]);
        
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        // Current inventory status
        $inventory = Dish::with('category')
            ->select('id', 'name', 'category_id', 'stock', 'price', 'created_at')
            ->get()
            ->map(function($dish) use ($startDate, $endDate) {
                // Calculate sales velocity
                $soldInPeriod = DB::table('order_items')
                    ->join('orders', 'order_items.order_id', '=', 'orders.id')
                    ->where('order_items.dish_id', $dish->id)
                    ->where('orders.status', 'completed')
                    ->whereBetween('orders.created_at', [$startDate, $endDate])
                    ->sum('order_items.quantity');
                
                $daysInPeriod = $startDate->diffInDays($endDate) + 1;
                $dailyVelocity = $daysInPeriod > 0 ? $soldInPeriod / $daysInPeriod : 0;
                $estimatedDaysUntilStockout = $dailyVelocity > 0 ? $dish->stock / $dailyVelocity : null;
                
                // Stock status
                $stockStatus = 'good';
                if ($dish->stock <= 5) $stockStatus = 'critical';
                elseif ($dish->stock <= 10) $stockStatus = 'low';
                elseif ($dish->stock <= 20) $stockStatus = 'warning';
                
                return [
                    'id' => $dish->id,
                    'name' => $dish->name,
                    'category' => $dish->category->name,
                    'current_stock' => $dish->stock,
                    'stock_status' => $stockStatus,
                    'sold_in_period' => $soldInPeriod,
                    'daily_velocity' => round($dailyVelocity, 2),
                    'estimated_stockout_days' => $estimatedDaysUntilStockout ? round($estimatedDaysUntilStockout) : null,
                    'stock_value' => $dish->stock * $dish->price,
                    'price' => $dish->price
                ];
            });
        
        // Stock alerts
        $stockAlerts = [
            'critical' => $inventory->where('stock_status', 'critical')->values(),
            'low' => $inventory->where('stock_status', 'low')->values(),
            'warning' => $inventory->where('stock_status', 'warning')->values()
        ];
        
        // Category-wise inventory
        $categoryInventory = $inventory->groupBy('category')
            ->map(function($items, $category) {
                return [
                    'category' => $category,
                    'total_items' => $items->count(),
                    'total_stock' => $items->sum('current_stock'),
                    'total_value' => $items->sum('stock_value'),
                    'avg_velocity' => $items->avg('daily_velocity'),
                    'critical_items' => $items->where('stock_status', 'critical')->count(),
                    'low_items' => $items->where('stock_status', 'low')->count()
                ];
            })->values();
        
        $summary = [
            'total_dishes' => $inventory->count(),
            'total_stock_units' => $inventory->sum('current_stock'),
            'total_stock_value' => $inventory->sum('stock_value'),
            'critical_stock_items' => $inventory->where('stock_status', 'critical')->count(),
            'low_stock_items' => $inventory->where('stock_status', 'low')->count(),
            'warning_stock_items' => $inventory->where('stock_status', 'warning')->count(),
            'avg_daily_velocity' => $inventory->avg('daily_velocity'),
            'period' => $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y')
        ];
        
        $data = compact('inventory', 'stockAlerts', 'categoryInventory', 'summary');
        
        if ($request->format === 'pdf') {
            // PDF generation temporarily disabled - DomPDF package needs to be installed
            return response()->json([
                'error' => 'PDF generation is currently unavailable. Please contact administrator to install the required packages.',
                'message' => 'Use CSV export as an alternative.'
            ], 503);
        }
        
        if ($request->format === 'csv') {
            return $this->exportInventoryCSV($inventory, $summary);
        }
        
        return view('super-admin.reports.inventory', $data);
    }
    
    public function performanceReport(Request $request)
    {
        // Provide defaults if parameters are missing
        $request->merge([
            'start_date' => $request->get('start_date', now()->subDays(30)->format('Y-m-d')),
            'end_date' => $request->get('end_date', now()->format('Y-m-d')),
            'format' => $request->get('format', 'view')
        ]);
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'format' => 'in:view,pdf,csv'
        ]);
        
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        // Order processing times
        $orderProcessingTimes = Order::whereBetween('created_at', [$startDate, $endDate])
            ->whereNotNull('updated_at')
            ->get()
            ->map(function($order) {
                $processingTime = $order->created_at->diffInMinutes($order->updated_at);
                return [
                    'order_id' => $order->id,
                    'status' => $order->status,
                    'processing_time_minutes' => $processingTime,
                    'created_at' => $order->created_at,
                    'updated_at' => $order->updated_at
                ];
            });
        
        // Order status distribution
        $statusDistribution = Order::whereBetween('created_at', [$startDate, $endDate])
            ->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status')
            ->toArray();
        
        // Peak hours analysis
        $peakHours = Order::whereBetween('created_at', [$startDate, $endDate])
            ->select(DB::raw('HOUR(created_at) as hour'), DB::raw('count(*) as order_count'), DB::raw('sum(total) as revenue'))
            ->groupBy(DB::raw('HOUR(created_at)'))
            ->orderBy('hour')
            ->get()
            ->map(function($hour) {
                return [
                    'hour' => $hour->hour . ':00',
                    'order_count' => $hour->order_count,
                    'revenue' => $hour->revenue,
                    'avg_order_value' => $hour->order_count > 0 ? $hour->revenue / $hour->order_count : 0
                ];
            });
        
        // Day of week analysis
        $dayOfWeekAnalysis = Order::whereBetween('created_at', [$startDate, $endDate])
            ->select(DB::raw('DAYNAME(created_at) as day_name'), DB::raw('DAYOFWEEK(created_at) as day_number'), 
                     DB::raw('count(*) as order_count'), DB::raw('sum(total) as revenue'))
            ->groupBy(DB::raw('DAYNAME(created_at)'), DB::raw('DAYOFWEEK(created_at)'))
            ->orderBy('day_number')
            ->get();
        
        // Customer satisfaction metrics (based on order completion rates)
        $satisfactionMetrics = [
            'completion_rate' => 0,
            'cancellation_rate' => 0,
            'average_processing_time' => $orderProcessingTimes->avg('processing_time_minutes') ?: 0,
            'fastest_processing_time' => $orderProcessingTimes->min('processing_time_minutes') ?: 0,
            'slowest_processing_time' => $orderProcessingTimes->max('processing_time_minutes') ?: 0
        ];
        
        $totalOrders = Order::whereBetween('created_at', [$startDate, $endDate])->count();
        if ($totalOrders > 0) {
            $completedOrders = Order::whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'completed')->count();
            $cancelledOrders = Order::whereBetween('created_at', [$startDate, $endDate])
                ->where('status', 'cancelled')->count();
                
            $satisfactionMetrics['completion_rate'] = ($completedOrders / $totalOrders) * 100;
            $satisfactionMetrics['cancellation_rate'] = ($cancelledOrders / $totalOrders) * 100;
        }
        
        $summary = [
            'total_orders_processed' => $totalOrders,
            'avg_processing_time' => round($satisfactionMetrics['average_processing_time'], 2),
            'completion_rate' => round($satisfactionMetrics['completion_rate'], 2),
            'cancellation_rate' => round($satisfactionMetrics['cancellation_rate'], 2),
            'peak_hour' => $peakHours->sortByDesc('order_count')->first()['hour'] ?? 'N/A',
            'busiest_day' => $dayOfWeekAnalysis->sortByDesc('order_count')->first()->day_name ?? 'N/A',
            'period' => $startDate->format('M d, Y') . ' - ' . $endDate->format('M d, Y')
        ];
        
        $data = compact('orderProcessingTimes', 'statusDistribution', 'peakHours', 'dayOfWeekAnalysis', 'satisfactionMetrics', 'summary');
        
        if ($request->format === 'pdf') {
            // PDF generation temporarily disabled - DomPDF package needs to be installed
            return response()->json([
                'error' => 'PDF generation is currently unavailable. Please contact administrator to install the required packages.',
                'message' => 'Use CSV export as an alternative.'
            ], 503);
        }
        
        if ($request->format === 'csv') {
            return $this->exportPerformanceCSV($orderProcessingTimes, $summary);
        }
        
        return view('super-admin.reports.performance', $data);
    }
    
    private function exportSalesCSV($orders, $summary)
    {
        $filename = 'sales-report-' . now()->format('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($orders, $summary) {
            $file = fopen('php://output', 'w');
            
            // Summary section
            fputcsv($file, ['Sales Report Summary']);
            fputcsv($file, ['Period', $summary['period']]);
            fputcsv($file, ['Total Orders', $summary['total_orders']]);
            fputcsv($file, ['Completed Orders', $summary['completed_orders']]);
            fputcsv($file, ['Pending Orders', $summary['pending_orders']]);
            fputcsv($file, ['Cancelled Orders', $summary['cancelled_orders']]);
            fputcsv($file, ['Total Revenue', '₦' . number_format($summary['total_revenue'], 2)]);
            fputcsv($file, ['Average Order Value', '₦' . number_format($summary['avg_order_value'], 2)]);
            fputcsv($file, ['Total Items Sold', $summary['total_items_sold']]);
            fputcsv($file, ['Conversion Rate', round($summary['conversion_rate'], 2) . '%']);
            fputcsv($file, []);
            
            // Orders data
            fputcsv($file, ['Order Details']);
            fputcsv($file, ['Order ID', 'Customer', 'Email', 'Status', 'Total', 'Items', 'Date']);
            
            foreach ($orders as $order) {
                fputcsv($file, [
                    $order->id,
                    $order->user->name ?? 'Guest',
                    $order->user->email ?? $order->guest_email,
                    $order->status,
                    '₦' . number_format($order->total, 2),
                    optional($order->items)->sum('quantity') ?? 0,
                    $order->created_at->format('Y-m-d H:i:s')
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    private function exportCustomersCSV($customers, $summary)
    {
        $filename = 'customer-report-' . now()->format('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($customers, $summary) {
            $file = fopen('php://output', 'w');
            
            // Summary section
            fputcsv($file, ['Customer Report Summary']);
            fputcsv($file, ['Period', $summary['period']]);
            fputcsv($file, ['Total Customers', $summary['total_customers']]);
            fputcsv($file, ['New Customers', $summary['new_customers']]);
            fputcsv($file, ['Total Revenue', '₦' . number_format($summary['total_revenue'], 2)]);
            fputcsv($file, ['Average Customer Value', '₦' . number_format($summary['avg_customer_value'], 2)]);
            fputcsv($file, ['Average Order Frequency', round($summary['avg_order_frequency'], 2)]);
            fputcsv($file, ['Customer Retention Rate', round($summary['customer_retention_rate'], 2) . '%']);
            fputcsv($file, []);
            
            // Customer data
            fputcsv($file, ['Customer Details']);
            fputcsv($file, ['ID', 'Name', 'Email', 'Phone', 'Registration Date', 'Total Orders', 'Total Spent', 'Avg Order Value', 'Lifetime Orders', 'Lifetime Value', 'Days Since Last Order']);
            
            foreach ($customers as $customer) {
                fputcsv($file, [
                    $customer['id'],
                    $customer['name'],
                    $customer['email'],
                    $customer['phone'],
                    $customer['registration_date']->format('Y-m-d'),
                    $customer['total_orders'],
                    '₦' . number_format($customer['total_spent'], 2),
                    '₦' . number_format($customer['avg_order_value'], 2),
                    $customer['lifetime_orders'],
                    '₦' . number_format($customer['lifetime_value'], 2),
                    $customer['days_since_last_order'] ?? 'N/A'
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    private function exportProductsCSV($products, $summary)
    {
        $filename = 'product-report-' . now()->format('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($products, $summary) {
            $file = fopen('php://output', 'w');
            
            // Summary section
            fputcsv($file, ['Product Report Summary']);
            fputcsv($file, ['Period', $summary['period']]);
            fputcsv($file, ['Total Products', $summary['total_products']]);
            fputcsv($file, ['Total Items Sold', $summary['total_items_sold']]);
            fputcsv($file, ['Total Revenue', '₦' . number_format($summary['total_revenue'], 2)]);
            fputcsv($file, ['Average Revenue per Product', '₦' . number_format($summary['avg_revenue_per_product'], 2)]);
            fputcsv($file, ['Low Stock Items', $summary['low_stock_items']]);
            fputcsv($file, ['Critical Stock Items', $summary['critical_stock_items']]);
            fputcsv($file, []);
            
            // Product data
            fputcsv($file, ['Product Details']);
            fputcsv($file, ['ID', 'Name', 'Category', 'Price', 'Current Stock', 'Total Sold', 'Total Revenue', 'Orders Count', 'First Sale', 'Last Sale']);
            
            foreach ($products as $product) {
                fputcsv($file, [
                    $product->id,
                    $product->name,
                    $product->category,
                    '₦' . number_format($product->price, 2),
                    $product->stock,
                    $product->total_sold,
                    '₦' . number_format($product->total_revenue, 2),
                    $product->orders_count,
                    $product->first_sale,
                    $product->last_sale
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    private function exportInventoryCSV($inventory, $summary)
    {
        $filename = 'inventory-report-' . now()->format('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($inventory, $summary) {
            $file = fopen('php://output', 'w');
            
            // Summary section
            fputcsv($file, ['Inventory Report Summary']);
            fputcsv($file, ['Report Date', now()->format('Y-m-d H:i:s')]);
            fputcsv($file, ['Total Dishes', $summary['total_dishes']]);
            fputcsv($file, ['Total Stock Units', $summary['total_stock_units']]);
            fputcsv($file, ['Total Stock Value', '₦' . number_format($summary['total_stock_value'], 2)]);
            fputcsv($file, ['Critical Stock Items', $summary['critical_stock_items']]);
            fputcsv($file, ['Low Stock Items', $summary['low_stock_items']]);
            fputcsv($file, ['Warning Stock Items', $summary['warning_stock_items']]);
            fputcsv($file, []);
            
            // Inventory data
            fputcsv($file, ['Inventory Details']);
            fputcsv($file, ['ID', 'Name', 'Category', 'Current Stock', 'Stock Status', 'Sold in Period', 'Daily Velocity', 'Estimated Stockout Days', 'Stock Value', 'Price']);
            
            foreach ($inventory as $item) {
                fputcsv($file, [
                    $item['id'],
                    $item['name'],
                    $item['category'],
                    $item['current_stock'],
                    ucfirst($item['stock_status']),
                    $item['sold_in_period'],
                    $item['daily_velocity'],
                    $item['estimated_stockout_days'] ?? 'N/A',
                    '₦' . number_format($item['stock_value'], 2),
                    '₦' . number_format($item['price'], 2)
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    private function exportPerformanceCSV($orderProcessingTimes, $summary)
    {
        $filename = 'performance-report-' . now()->format('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($orderProcessingTimes, $summary) {
            $file = fopen('php://output', 'w');
            
            // Summary section
            fputcsv($file, ['Performance Report Summary']);
            fputcsv($file, ['Period', $summary['period']]);
            fputcsv($file, ['Total Orders Processed', $summary['total_orders_processed']]);
            fputcsv($file, ['Average Processing Time (minutes)', $summary['avg_processing_time']]);
            fputcsv($file, ['Completion Rate (%)', $summary['completion_rate']]);
            fputcsv($file, ['Cancellation Rate (%)', $summary['cancellation_rate']]);
            fputcsv($file, ['Peak Hour', $summary['peak_hour']]);
            fputcsv($file, ['Busiest Day', $summary['busiest_day']]);
            fputcsv($file, []);
            
            // Order processing data
            fputcsv($file, ['Order Processing Details']);
            fputcsv($file, ['Order ID', 'Status', 'Processing Time (minutes)', 'Created At', 'Updated At']);
            
            foreach ($orderProcessingTimes as $order) {
                fputcsv($file, [
                    $order['order_id'],
                    $order['status'],
                    $order['processing_time_minutes'],
                    $order['created_at']->format('Y-m-d H:i:s'),
                    $order['updated_at']->format('Y-m-d H:i:s')
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
}