<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Dish;
use App\Models\Order;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SuperAdminDashboardController extends Controller
{
    public function index()
    {
        // Key metrics
        $totalUsers = User::count();
        $totalAdmins = User::role(['Admin', 'Super Admin'])->count();
        $totalDishes = Dish::count();
        $totalOrders = Order::count();
    $totalRevenue = Order::where('status', 'completed')->sum('total');
        
        // Recent activity
        $recentUsers = User::latest()->take(5)->get();
        $recentOrders = Order::with(['user', 'items.dish'])->latest()->take(5)->get();
        
        // Growth metrics (last 30 days vs previous 30 days)
        $currentPeriod = Carbon::now()->subDays(30);
        $previousPeriod = Carbon::now()->subDays(60);
        
        $currentUsers = User::where('created_at', '>=', $currentPeriod)->count();
        $previousUsers = User::whereBetween('created_at', [$previousPeriod, $currentPeriod])->count();
        $userGrowth = $previousUsers > 0 ? (($currentUsers - $previousUsers) / $previousUsers) * 100 : 0;
        
        $currentRevenue = Order::where('status', 'completed')
            ->where('created_at', '>=', $currentPeriod)
            ->sum('total');
        $previousRevenue = Order::where('status', 'completed')
            ->whereBetween('created_at', [$previousPeriod, $currentPeriod])
            ->sum('total');
        $revenueGrowth = $previousRevenue > 0 ? (($currentRevenue - $previousRevenue) / $previousRevenue) * 100 : 0;
        
        // Chart data - Daily orders for last 30 days
    $dailyOrders = Order::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as count'),
        DB::raw('SUM(total) as revenue')
            )
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->groupBy('date')
            ->orderBy('date')
            ->get();
        
        // Top performing dishes
        $topDishes = DB::table('order_items')
            ->join('dishes', 'order_items.dish_id', '=', 'dishes.id')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('orders.status', 'completed')
            ->select(
                'dishes.name',
                DB::raw('SUM(order_items.quantity) as total_sold'),
                DB::raw('SUM(order_items.quantity * order_items.price) as total_revenue')
            )
            ->groupBy('dishes.id', 'dishes.name')
            ->orderByDesc('total_sold')
            ->take(10)
            ->get();
        
        // User role distribution
        $roleDistribution = User::select(
                DB::raw('COALESCE(model_has_roles.role_id, 0) as role_id'),
                DB::raw('COUNT(*) as count')
            )
            ->leftJoin('model_has_roles', function($join) {
                $join->on('users.id', '=', 'model_has_roles.model_id')
                     ->where('model_has_roles.model_type', '=', User::class);
            })
            ->groupBy('role_id')
            ->get();
        
        return view('super-admin.dashboard', compact(
            'totalUsers', 'totalAdmins', 'totalDishes', 'totalOrders', 'totalRevenue',
            'recentUsers', 'recentOrders', 'userGrowth', 'revenueGrowth',
            'dailyOrders', 'topDishes', 'roleDistribution'
        ));
    }
}