<?php

namespace App\Jobs;

use App\Models\Dish;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;

/**
 * Generates responsive variants (AVIF / WebP / JPEG) for a single dish image upload
 * and merges into the public/assets/image-manifest.json file.
 *
 * This is a lightweight mirror of the bulk script (scripts/generate-variants.js)
 * focused only on dish images, using PHP + Intervention via GD/Image functions is possible
 * but we rely on the 'spatie/image' or native GD? To keep dependencies minimal we
 * just create resized JPEG fallbacks if the 'imagick' extension is missing.
 *
 * NOTE: For full AVIF/WebP quality you can still run the Node script; this job ensures
 * an immediate usable set of sizes so pages don't reference huge originals.
 */
class GenerateDishImageVariants implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $dishId;
    public string $relativePath; // e.g. dishes/abc123.jpg (storage/public)

    public $tries = 2;
    public $timeout = 120;

    protected array $widths = [200,400,600,800,1200,1600];

    public function __construct(int $dishId, string $relativePath)
    {
        $this->dishId = $dishId;
        $this->relativePath = ltrim($relativePath, '/');
    }

    public function handle(): void
    {
        $dish = Dish::find($this->dishId);
        if (!$dish) return;
        $publicDisk = Storage::disk('public');
        if (!$publicDisk->exists($this->relativePath)) return;

        $sourceFull = $publicDisk->path($this->relativePath);
        $baseName = pathinfo($sourceFull, PATHINFO_FILENAME);
        $outDir = public_path('assets/dishes');
        if (!is_dir($outDir)) mkdir($outDir, 0775, true);

        // We'll try to leverage GD functions (always present on most PHP builds) for quick JPEG downsizes.
        $imgInfo = @getimagesize($sourceFull);
        if (!$imgInfo) return;
        [$origW,$origH] = [$imgInfo[0], $imgInfo[1]];
        $mime = $imgInfo['mime'] ?? '';
        $createFn = match($mime) {
            'image/jpeg' => 'imagecreatefromjpeg',
            'image/png' => 'imagecreatefrompng',
            'image/webp' => function($p){ return function_exists('imagecreatefromwebp') ? imagecreatefromwebp($p) : null; },
            default => null,
        };
        if (!$createFn || !is_callable($createFn)) return; // unsupported format
        $im = @($createFn)($sourceFull);
        if (!$im) return;

        $variants = [];
        foreach ($this->widths as $w) {
            if ($w >= $origW) continue; // skip upscaling large sizes
            $ratio = $origH / $origW;
            $h = (int) round($w * $ratio);
            $canvas = imagecreatetruecolor($w, $h);
            imagecopyresampled($canvas, $im, 0,0,0,0, $w,$h, $origW,$origH);
            $jpgRel = 'assets/dishes/'.$baseName.'-'.$w.'.jpg';
            $jpgPath = public_path($jpgRel);
            imagejpeg($canvas, $jpgPath, 80);
            imagedestroy($canvas);
            $variants[] = [ 'width' => $w, 'jpg' => $jpgRel ];
        }

        // LQIP tiny
        $lqipRel = null;
        $lqipW = 24;
        if ($origW > 0) {
            $lqipRatio = $origH / $origW;
            $lqipH = (int) round($lqipW * $lqipRatio);
            $lqipCanvas = imagecreatetruecolor($lqipW, $lqipH);
            imagecopyresampled($lqipCanvas, $im,0,0,0,0,$lqipW,$lqipH,$origW,$origH);
            $lqipRel = 'assets/dishes/'.$baseName.'-lqip.jpg';
            imagejpeg($lqipCanvas, public_path($lqipRel), 35);
            imagedestroy($lqipCanvas);
        }

        imagedestroy($im);

        // Merge into manifest (create if absent)
        $manifestFile = public_path('assets/image-manifest.json');
        $manifest = [ 'generatedAt' => now()->toIso8601String(), 'slider' => [], 'dishes' => [] ];
        if (is_file($manifestFile)) {
            try { $json = file_get_contents($manifestFile); $decoded = json_decode($json, true); if (is_array($decoded)) $manifest = array_merge($manifest, $decoded); } catch (\Throwable $e) {}
        }
        // Remove existing entry with same base if exists
        $manifest['dishes'] = array_values(array_filter($manifest['dishes'], fn($e)=> ($e['base']??null) !== $baseName));
        $manifest['dishes'][] = [
            'type' => 'dish',
            'base' => $baseName,
            'variants' => $variants,
            'thumb' => null,
            'lqip' => $lqipRel,
            'original' => 'storage/dishes/'.basename($this->relativePath),
            'partial' => true, // marks that WebP/AVIF not yet generated or GD missing
        ];
        // Update generatedAt to reflect change
        $manifest['generatedAt'] = now()->toIso8601String();
        file_put_contents($manifestFile, json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }
}
