<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Dish extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [ 'category_id','name','description','price','stock','low_stock_threshold','velocity_7d','velocity_30d','safety_stock','reorder_point','projected_runway_days','velocity_updated_at','image','vegetarian','halal','gluten_free' ];

    protected $casts = [ 
        'price' => 'decimal:2', 
        'stock' => 'integer', 
        'low_stock_threshold' => 'integer', 
        'velocity_7d' => 'decimal:2',
        'velocity_30d' => 'decimal:2',
        'safety_stock' => 'integer',
        'reorder_point' => 'integer',
        'projected_runway_days' => 'decimal:1',
        'velocity_updated_at' => 'datetime',
        'vegetarian' => 'boolean', 
        'halal' => 'boolean', 
        'gluten_free' => 'boolean' 
    ];

    public function category() { return $this->belongsTo(Category::class); }

    /**
     * Attempt to find an asset image for a given dish name under public/assets/images.
     * Returns a relative public path like 'assets/images/egusi.jpg' if found.
     */
    public static function findAssetImageForName(string $name): ?string
    {
        $base = mb_strtolower($name);
        $base = str_replace(['&','/','\\'], [' and ',' ',' '], $base);
        $base = preg_replace('/\s+/', ' ', $base);
        $slug = \Illuminate\Support\Str::slug($base, '-');
        $under = preg_replace('/[^a-z0-9]+/', '_', $base);
        $under = trim($under, '_');
        $compact = preg_replace('/[^a-z0-9]+/','', $base);
        $candidates = array_values(array_unique([
            $slug,
            $under,
            $compact,
            explode('-', $slug)[0] ?? $slug,
        ]));
        $exts = ['jpg','jpeg','png','webp','avif'];
        foreach ($candidates as $stem) {
            foreach ($exts as $ext) {
                $rel = 'assets/images/'.$stem.'.'.$ext;
                if (is_file(public_path($rel))) return $rel;
            }
        }
        return null;
    }

    /** Assign an image automatically from assets if empty and a match exists. */
    public function ensureImageAssigned(): void
    {
        if ($this->image) return;
        $match = static::findAssetImageForName((string) $this->name);
        if ($match) {
            $this->image = $match; // store relative path under public
        }
    }

    protected static function booted()
    {
        static::saving(function (Dish $dish) {
            // Auto-assign an image from assets if none provided
            $dish->ensureImageAssigned();
        });

        // After the model is saved, if the image has changed and points to a storage-backed file,
        // generate responsive variants and a tiny LQIP synchronously.
        static::saved(function (Dish $dish) {
            try {
                if ($dish->wasChanged('image')) {
                    $img = (string) $dish->image;
                    if ($img !== ''
                        && !str_starts_with($img, 'assets/')
                        && !str_starts_with($img, '/assets/')
                        && !str_starts_with($img, 'http://')
                        && !str_starts_with($img, 'https://')
                        && (str_starts_with($img, 'dishes/') || !str_starts_with($img, 'storage/'))
                    ) {
                        // $img is expected to be a relative path on the 'public' disk, e.g., 'dishes/abc.jpg'
                        (new \App\Support\Images\DishImageVariants())->generate($img);
                    }
                }
            } catch (\Throwable $e) {
                // Silently ignore variant generation failures; UI can fall back to original
            }
        });
    }

    /**
     * Return a publicly accessible URL for this dish's image, handling:
     *  - Absolute remote URLs
     *  - Seeded static assets under public/assets/images
     *  - Storage-backed images (requires storage:link)
     *  - Fallback copied originals (assets/dishes/originals)
     */
    public function getPublicImageUrlAttribute(): ?string
    {
        $img = $this->image;
        if (!$img) return null;
        if (str_starts_with($img, 'http://') || str_starts_with($img, 'https://')) {
            return $img; // absolute remote URL
        }
        // If it already points under public assets, just return it
        if (str_starts_with($img, 'assets/')) {
            return '/'.ltrim($img, '/'); // root-relative under public/
        }

        // Common storage conventions
        $candidates = [];
        // Direct storage path provided (e.g., dishes/xxx.jpg)
        $candidates[] = 'storage/'.ltrim($img, '/');
        // Explicit dishes directory inside storage symlink
        $candidates[] = 'storage/dishes/'.ltrim($img, '/');
        // In case the image field already includes 'dishes/' remove duplication
        $candidates[] = 'storage/'.ltrim(preg_replace('#^/?storage/#','',$img), '/');

        // Legacy public locations (copied assets)
        $candidates[] = 'assets/dishes/'.basename($img);
        $candidates[] = 'assets/images/'.basename($img);
        $candidates[] = 'assets/dishes/originals/'.basename($img);

        foreach ($candidates as $rel) {
            if (is_file(public_path($rel))) {
                return '/'.ltrim($rel, '/'); // root-relative to avoid APP_URL port mismatches
            }
        }

        return null; // No resolvable file; caller should show placeholder
    }

    /** Base candidate for image manifest lookups (without extension) */
    public function getImageBaseAttribute(): ?string
    {
        if (!$this->image) return null;
        $name = pathinfo($this->image, PATHINFO_FILENAME);
        // Expect hashed name length >= 20 for variant generation eligibility
        if ($name && strlen($name) >= 20) return $name;
        return null;
    }

    /**
     * Check if the dish stock is considered low based on its threshold.
     */
    public function isLowStock(): bool
    {
        return $this->stock <= $this->low_stock_threshold;
    }

    /**
     * Check if the dish is out of stock.
     */
    public function isOutOfStock(): bool
    {
        return $this->stock <= 0;
    }

    /**
     * Check if stock is below safety stock level.
     */
    public function isBelowSafetyStock(): bool
    {
        return $this->safety_stock > 0 && $this->stock < $this->safety_stock;
    }

    /**
     * Check if stock is at or below reorder point.
     */
    public function needsReorder(): bool
    {
        return $this->stock <= $this->reorder_point;
    }

    /**
     * Get projected days until stockout based on current velocity.
     */
    public function getProjectedStockoutDays(): ?float
    {
        if ($this->velocity_7d <= 0) {
            return null; // No consumption, infinite runway
        }
        
        return round($this->stock / $this->velocity_7d, 1);
    }

    /**
     * Check if dish is predicted to stock out within given days.
     */
    public function willStockOutWithin(int $days): bool
    {
        $projectedDays = $this->getProjectedStockoutDays();
        return $projectedDays !== null && $projectedDays <= $days;
    }

    /**
     * Relationship to inventory alerts.
     */
    public function inventoryAlerts()
    {
        return $this->hasMany(InventoryAlert::class);
    }

    /**
     * Get active inventory alerts.
     */
    public function activeAlerts()
    {
        return $this->inventoryAlerts()->active();
    }

    /**
     * Get inventory stats for this dish.
     */
    public function inventoryStats()
    {
        return $this->hasOne(InventoryStats::class);
    }
}
