<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class InventoryAlert extends Model
{
    use HasFactory;

    protected $fillable = [
        'dish_id',
        'type',
        'severity',
        'status',
        'title',
        'message',
        'metadata',
        'projected_stockout_days',
        'current_stock',
        'current_velocity',
        'acknowledged_by',
        'acknowledged_at',
        'acknowledgment_note',
    ];

    protected $casts = [
        'metadata' => 'array',
        'projected_stockout_days' => 'decimal:1',
        'current_velocity' => 'decimal:2',
        'acknowledged_at' => 'datetime',
    ];

    // Alert types
    public const TYPE_LOW_STOCK = 'low_stock';
    public const TYPE_OUT_OF_STOCK = 'out_of_stock';
    public const TYPE_PROJECTED_STOCKOUT = 'projected_stockout';
    public const TYPE_SAFETY_STOCK_BREACH = 'safety_stock_breach';

    // Severity levels
    public const SEVERITY_LOW = 'low';
    public const SEVERITY_MEDIUM = 'medium';
    public const SEVERITY_HIGH = 'high';
    public const SEVERITY_CRITICAL = 'critical';

    // Status values
    public const STATUS_ACTIVE = 'active';
    public const STATUS_ACKNOWLEDGED = 'acknowledged';
    public const STATUS_DISMISSED = 'dismissed';
    public const STATUS_RESOLVED = 'resolved';

    /**
     * Relationships
     */
    public function dish(): BelongsTo
    {
        return $this->belongsTo(Dish::class);
    }

    public function acknowledgedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'acknowledged_by');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    public function scopeHighPriority($query)
    {
        return $query->whereIn('severity', [self::SEVERITY_HIGH, self::SEVERITY_CRITICAL]);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Helper methods
     */
    public function isActive(): bool
    {
        return $this->status === self::STATUS_ACTIVE;
    }

    public function isAcknowledged(): bool
    {
        return $this->status === self::STATUS_ACKNOWLEDGED;
    }

    public function isCritical(): bool
    {
        return $this->severity === self::SEVERITY_CRITICAL;
    }

    public function acknowledge(User $user, ?string $note = null): bool
    {
        return $this->update([
            'status' => self::STATUS_ACKNOWLEDGED,
            'acknowledged_by' => $user->id,
            'acknowledged_at' => now(),
            'acknowledgment_note' => $note,
        ]);
    }

    public function dismiss(): bool
    {
        return $this->update(['status' => self::STATUS_DISMISSED]);
    }

    public function resolve(): bool
    {
        return $this->update(['status' => self::STATUS_RESOLVED]);
    }

    /**
     * Get severity color for UI
     */
    public function getSeverityColorAttribute(): string
    {
        return match($this->severity) {
            self::SEVERITY_LOW => 'gray',
            self::SEVERITY_MEDIUM => 'yellow',
            self::SEVERITY_HIGH => 'orange',
            self::SEVERITY_CRITICAL => 'red',
            default => 'gray',
        };
    }

    /**
     * Get type icon for UI
     */
    public function getTypeIconAttribute(): string
    {
        return match($this->type) {
            self::TYPE_LOW_STOCK => 'alert-triangle',
            self::TYPE_OUT_OF_STOCK => 'alert-circle',
            self::TYPE_PROJECTED_STOCKOUT => 'clock',
            self::TYPE_SAFETY_STOCK_BREACH => 'shield-alert',
            default => 'alert-triangle',
        };
    }

    /**
     * Get severity badge for display
     */
    public function getSeverityBadge(): string
    {
        return match($this->severity) {
            self::SEVERITY_CRITICAL => '🔴 Critical',
            self::SEVERITY_HIGH => '🟠 High',
            self::SEVERITY_MEDIUM => '🟡 Medium',
            self::SEVERITY_LOW => '🟢 Low',
            default => '⚪ Unknown',
        };
    }

    /**
     * Get type label for display
     */
    public function getTypeLabel(): string
    {
        return match($this->type) {
            self::TYPE_LOW_STOCK => 'Low Stock',
            self::TYPE_OUT_OF_STOCK => 'Out of Stock',
            self::TYPE_PROJECTED_STOCKOUT => 'Projected Stockout',
            self::TYPE_SAFETY_STOCK_BREACH => 'Safety Stock Breach',
            default => 'Unknown',
        };
    }

    /**
     * Get status badge for display
     */
    public function getStatusBadge(): string
    {
        return match($this->status) {
            self::STATUS_ACTIVE => '🟢 Active',
            self::STATUS_ACKNOWLEDGED => '🟡 Acknowledged',
            self::STATUS_RESOLVED => '✅ Resolved',
            self::STATUS_DISMISSED => '❌ Dismissed',
            default => '⚪ Unknown',
        };
    }
}
