<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class InventorySnapshot extends Model
{
    use HasFactory;

    const TYPE_DISH = 'dish';
    const TYPE_GLOBAL = 'global';

    protected $fillable = [
        'snapshot_date',
        'dish_id',
        'type',
        'on_hand',
        'allocated',
        'available',
        'sales_qty',
        'sales_value',
        'adjustments_qty',
        'movements_count',
        'velocity_7d',
        'velocity_30d',
        'projected_runway_days',
        'inventory_value',
        'average_sale_price',
        'was_out_of_stock',
        'was_low_stock',
        'had_stockout_alert',
        'active_alerts_count',
        'fill_rate',
        'stockout_events',
        'total_dishes',
        'dishes_with_stock',
        'dishes_low_stock',
        'dishes_out_of_stock',
        'total_inventory_value',
        'avg_runway_days',
    ];

    protected $casts = [
        'snapshot_date' => 'date',
        'velocity_7d' => 'decimal:2',
        'velocity_30d' => 'decimal:2',
        'projected_runway_days' => 'decimal:1',
        'sales_value' => 'decimal:2',
        'inventory_value' => 'decimal:2',
        'average_sale_price' => 'decimal:2',
        'fill_rate' => 'decimal:2',
        'total_inventory_value' => 'decimal:2',
        'avg_runway_days' => 'decimal:1',
        'was_out_of_stock' => 'boolean',
        'was_low_stock' => 'boolean',
        'had_stockout_alert' => 'boolean',
    ];

    /**
     * Get the dish that owns the snapshot.
     */
    public function dish(): BelongsTo
    {
        return $this->belongsTo(Dish::class);
    }

    /**
     * Scope for dish-specific snapshots
     */
    public function scopeDishSnapshots($query)
    {
        return $query->where('type', self::TYPE_DISH);
    }

    /**
     * Scope for global snapshots
     */
    public function scopeGlobalSnapshots($query)
    {
        return $query->where('type', self::TYPE_GLOBAL);
    }

    /**
     * Scope for date range
     */
    public function scopeDateRange($query, Carbon $startDate, Carbon $endDate)
    {
        return $query->whereBetween('snapshot_date', [$startDate->format('Y-m-d'), $endDate->format('Y-m-d')]);
    }

    /**
     * Scope for recent snapshots
     */
    public function scopeRecent($query, int $days = 30)
    {
        return $query->where('snapshot_date', '>=', now()->subDays($days)->format('Y-m-d'));
    }

    /**
     * Scope for snapshots with sales activity
     */
    public function scopeWithSales($query)
    {
        return $query->where('sales_qty', '>', 0);
    }

    /**
     * Scope for out of stock events
     */
    public function scopeOutOfStockEvents($query)
    {
        return $query->where('was_out_of_stock', true);
    }

    /**
     * Scope for low stock events
     */
    public function scopeLowStockEvents($query)
    {
        return $query->where('was_low_stock', true);
    }

    /**
     * Get daily sales trend for a dish
     */
    public static function getDailySalesTrend(int $dishId, int $days = 30): array
    {
        return self::dishSnapshots()
            ->where('dish_id', $dishId)
            ->recent($days)
            ->orderBy('snapshot_date')
            ->pluck('sales_qty', 'snapshot_date')
            ->toArray();
    }

    /**
     * Get velocity trend for a dish
     */
    public static function getVelocityTrend(int $dishId, int $days = 30): array
    {
        return self::dishSnapshots()
            ->where('dish_id', $dishId)
            ->recent($days)
            ->orderBy('snapshot_date')
            ->get(['snapshot_date', 'velocity_7d', 'velocity_30d'])
            ->map(function ($snapshot) {
                return [
                    'date' => $snapshot->snapshot_date->format('Y-m-d'),
                    'velocity_7d' => (float) $snapshot->velocity_7d,
                    'velocity_30d' => (float) $snapshot->velocity_30d,
                ];
            })
            ->toArray();
    }

    /**
     * Get stock level trend for a dish
     */
    public static function getStockTrend(int $dishId, int $days = 30): array
    {
        return self::dishSnapshots()
            ->where('dish_id', $dishId)
            ->recent($days)
            ->orderBy('snapshot_date')
            ->get(['snapshot_date', 'on_hand', 'available', 'allocated'])
            ->map(function ($snapshot) {
                return [
                    'date' => $snapshot->snapshot_date->format('Y-m-d'),
                    'on_hand' => $snapshot->on_hand,
                    'available' => $snapshot->available,
                    'allocated' => $snapshot->allocated,
                ];
            })
            ->toArray();
    }

    /**
     * Get global inventory health trend
     */
    public static function getHealthTrend(int $days = 30): array
    {
        return self::globalSnapshots()
            ->recent($days)
            ->orderBy('snapshot_date')
            ->get(['snapshot_date', 'dishes_with_stock', 'dishes_low_stock', 'dishes_out_of_stock', 'total_dishes'])
            ->map(function ($snapshot) {
                $healthScore = 100;
                if ($snapshot->total_dishes > 0) {
                    $outOfStockRatio = $snapshot->dishes_out_of_stock / $snapshot->total_dishes;
                    $lowStockRatio = $snapshot->dishes_low_stock / $snapshot->total_dishes;
                    $healthScore = 100 - ($outOfStockRatio * 50) - ($lowStockRatio * 25);
                }
                
                return [
                    'date' => $snapshot->snapshot_date->format('Y-m-d'),
                    'health_score' => round(max(0, $healthScore), 1),
                    'dishes_with_stock' => $snapshot->dishes_with_stock,
                    'dishes_low_stock' => $snapshot->dishes_low_stock,
                    'dishes_out_of_stock' => $snapshot->dishes_out_of_stock,
                    'total_dishes' => $snapshot->total_dishes,
                ];
            })
            ->toArray();
    }

    /**
     * Calculate stockout frequency for a dish
     */
    public static function getStockoutFrequency(int $dishId, int $days = 30): array
    {
        $snapshots = self::dishSnapshots()
            ->where('dish_id', $dishId)
            ->recent($days)
            ->get();

        $totalDays = $snapshots->count();
        $stockoutDays = $snapshots->where('was_out_of_stock', true)->count();
        $lowStockDays = $snapshots->where('was_low_stock', true)->count();

        return [
            'total_days' => $totalDays,
            'stockout_days' => $stockoutDays,
            'low_stock_days' => $lowStockDays,
            'stockout_rate' => $totalDays > 0 ? round(($stockoutDays / $totalDays) * 100, 2) : 0,
            'low_stock_rate' => $totalDays > 0 ? round(($lowStockDays / $totalDays) * 100, 2) : 0,
            'availability_rate' => $totalDays > 0 ? round((($totalDays - $stockoutDays) / $totalDays) * 100, 2) : 0,
        ];
    }

    /**
     * Get top performing dishes by sales
     */
    public static function getTopPerformers(Carbon $date, int $limit = 10): array
    {
        return self::dishSnapshots()
            ->where('snapshot_date', $date->format('Y-m-d'))
            ->with('dish:id,name')
            ->where('sales_qty', '>', 0)
            ->orderByDesc('sales_value')
            ->limit($limit)
            ->get()
            ->map(function ($snapshot) {
                return [
                    'dish_name' => $snapshot->dish->name ?? 'Unknown',
                    'sales_qty' => $snapshot->sales_qty,
                    'sales_value' => (float) $snapshot->sales_value,
                    'velocity_7d' => (float) $snapshot->velocity_7d,
                ];
            })
            ->toArray();
    }

    /**
     * Get comparative performance (vs previous period)
     */
    public static function getComparativePerformance(int $dishId, Carbon $date): array
    {
        $current = self::dishSnapshots()
            ->where('dish_id', $dishId)
            ->where('snapshot_date', $date->format('Y-m-d'))
            ->first();

        $previous = self::dishSnapshots()
            ->where('dish_id', $dishId)
            ->where('snapshot_date', $date->copy()->subDay()->format('Y-m-d'))
            ->first();

        if (!$current || !$previous) {
            return [];
        }

        return [
            'sales_change' => $current->sales_qty - $previous->sales_qty,
            'sales_change_percent' => $previous->sales_qty > 0 
                ? round((($current->sales_qty - $previous->sales_qty) / $previous->sales_qty) * 100, 2)
                : null,
            'stock_change' => $current->on_hand - $previous->on_hand,
            'velocity_change' => (float) $current->velocity_7d - (float) $previous->velocity_7d,
        ];
    }
}
