<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class InventoryStats extends Model
{
    use HasFactory;

    protected $fillable = [
        'dish_id',
        'location_id',
        'on_hand',
        'allocated',
        'available',
        'total_sales',
        'total_adjustments',
        'total_movements',
        'velocity_7d',
        'velocity_30d',
        'projected_runway_days',
        'total_sales_value',
        'average_sale_price',
        'inventory_value',
        'stockout_events',
        'low_stock_events',
        'reorder_events',
        'last_movement_at',
        'last_sale_at',
        'last_stockout_at',
        'stats_calculated_at',
    ];

    protected $casts = [
        'velocity_7d' => 'decimal:2',
        'velocity_30d' => 'decimal:2',
        'projected_runway_days' => 'decimal:1',
        'total_sales_value' => 'decimal:2',
        'average_sale_price' => 'decimal:2',
        'inventory_value' => 'decimal:2',
        'last_movement_at' => 'datetime',
        'last_sale_at' => 'datetime',
        'last_stockout_at' => 'datetime',
        'stats_calculated_at' => 'datetime',
    ];

    /**
     * Get the dish that owns the stats.
     */
    public function dish(): BelongsTo
    {
        return $this->belongsTo(Dish::class);
    }

    /**
     * Location for per-location stats (nullable for global stats)
     */
    public function location(): BelongsTo
    {
        return $this->belongsTo(Location::class);
    }

    /**
     * Scope for available stock filtering
     */
    public function scopeAvailable($query, $minQuantity = 1)
    {
        return $query->where('available', '>=', $minQuantity);
    }

    /**
     * Scope for low stock items
     */
    public function scopeLowStock($query)
    {
        return $query->whereHas('dish', function ($q) {
            $q->whereColumn('stock', '<=', 'low_stock_threshold');
        });
    }

    /**
     * Scope for out of stock items
     */
    public function scopeOutOfStock($query)
    {
        return $query->where('on_hand', '<=', 0);
    }

    /**
     * Scope for high velocity items
     */
    public function scopeHighVelocity($query, $threshold = 1.0)
    {
        return $query->where('velocity_7d', '>=', $threshold);
    }

    /**
     * Scope for fast-moving items (top velocity)
     */
    public function scopeFastMoving($query, $limit = 10)
    {
        return $query->orderByDesc('velocity_7d')->limit($limit);
    }

    /**
     * Scope for slow-moving items (low velocity)
     */
    public function scopeSlowMoving($query, $threshold = 0.1)
    {
        return $query->where('velocity_7d', '<=', $threshold)
                     ->where('velocity_7d', '>', 0);
    }

    /**
     * Scope for items with recent activity
     */
    public function scopeRecentActivity($query, $days = 7)
    {
        return $query->where('last_movement_at', '>=', now()->subDays($days));
    }

    /**
     * Check if item is critically low stock
     */
    public function isCriticallyLow(): bool
    {
        return $this->projected_runway_days <= 3 && $this->projected_runway_days > 0;
    }

    /**
     * Check if item needs reordering
     */
    public function needsReordering(): bool
    {
        // Needs reordering if we have reorder point data and we're below it
        return $this->dish && 
               $this->dish->reorder_point > 0 && 
               $this->on_hand <= $this->dish->reorder_point;
    }

    /**
     * Calculate inventory turnover ratio
     */
    public function calculateTurnoverRatio(int $days = 30): ?float
    {
        if ($this->inventory_value <= 0) {
            return null;
        }

        // Calculate COGS approximation from sales
        $dailySalesValue = $this->velocity_7d * ($this->average_sale_price ?? 0);
        $periodCogs = $dailySalesValue * $days;

        return $periodCogs > 0 ? $periodCogs / $this->inventory_value : 0;
    }

    /**
     * Get performance score (0-100)
     */
    public function getPerformanceScore(): int
    {
        $score = 100;

        // Deduct for stockouts
        $score -= min(50, $this->stockout_events * 10);

        // Deduct for low velocity (dead stock)
        if ($this->velocity_7d <= 0) {
            $score -= 20;
        } elseif ($this->velocity_7d < 0.1) {
            $score -= 10;
        }

        // Deduct for excessive stock (poor turnover)
        if ($this->projected_runway_days > 90) {
            $score -= 15;
        } elseif ($this->projected_runway_days > 60) {
            $score -= 10;
        }

        // Bonus for consistent availability
        if ($this->stockout_events === 0 && $this->on_hand > 0) {
            $score += 5;
        }

        return max(0, min(100, $score));
    }

    /**
     * Get status indicator
     */
    public function getStatusIndicator(): array
    {
        if ($this->on_hand <= 0) {
            return [
                'status' => 'out_of_stock',
                'color' => 'red',
                'label' => 'Out of Stock',
                'icon' => 'alert-circle'
            ];
        }

        if ($this->isCriticallyLow()) {
            return [
                'status' => 'critical',
                'color' => 'red',
                'label' => 'Critical Low',
                'icon' => 'alert-triangle'
            ];
        }

        if ($this->needsReordering()) {
            return [
                'status' => 'reorder',
                'color' => 'orange',
                'label' => 'Needs Reorder',
                'icon' => 'shopping-cart'
            ];
        }

        if ($this->velocity_7d <= 0) {
            return [
                'status' => 'stagnant',
                'color' => 'gray',
                'label' => 'No Movement',
                'icon' => 'pause-circle'
            ];
        }

        return [
            'status' => 'healthy',
            'color' => 'green',
            'label' => 'Healthy',
            'icon' => 'check-circle'
        ];
    }
}
