<?php
// VIEW CONTENT REQUEST PLACEHOLDER

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Enums\OrderState;
use App\Support\Orders\OrderLifecycle;

/**
 * Order Model
 * NOTE: This file was reconstructed; ensure fields align with current migration.
 */
class Order extends Model
{
    use HasFactory;

    /**
     * Canonical list of statuses (used in admin UI, printing, etc.).
     * Keep order to reflect typical lifecycle.
     */
    public const STATUSES = [
        'Received',
        'In Kitchen',
        'Ready',
        'Out for Delivery',
        'Delivered',
        'Cancelled',
    ];

    /**
     * Canonical state machine (internal progression). Distinct from legacy display status.
     * States are lower_snake_case; map to statuses in UI as needed.
     */
    /**
     * Return enum state object (fallback legacy mapping when null).
     */
    public function stateEnum(): OrderState
    {
        if ($this->state) return OrderState::from($this->state);
        return (new OrderLifecycle())->currentState($this);
    }

    /** Legacy display status derived from enum if status column becomes stale. */
    public function getDisplayStatusAttribute(): string
    {
        return $this->stateEnum()->toDisplayStatus();
    }

    protected $fillable = [
        'user_id','stay_id','customer_address_id','subtotal','delivery_fee','discount','service_charge','total','status','state','delivery_mode','ordered_by_guest','guest_room_number','delivery_address','delivery_address_verified','delivery_latitude','delivery_longitude','delivery_distance_km','verification_log','payment_method','tip','payment_reference','room_number'
    ];

    protected $casts = [
        'delivery_distance_km' => 'float',
        'delivery_address_verified' => 'boolean',
        'tip' => 'float',
    ];

    public function user(): BelongsTo { return $this->belongsTo(User::class); }
    public function items(): HasMany { return $this->hasMany(OrderItem::class); }
    public function customerAddress(): BelongsTo { return $this->belongsTo(CustomerAddress::class); }
    public function statusEvents(): HasMany { return $this->hasMany(OrderStatusEvent::class); }

    /**
     * Derived human-friendly delivery destination.
     */
    public function getDisplayDeliveryAddressAttribute(): ?string
    {
        if ($this->delivery_address) return $this->delivery_address;
        if ($this->delivery_mode === 'room') {
            if ($this->guest_room_number) return 'Room '.$this->guest_room_number;
            if ($this->room_number) return 'Room '.$this->room_number;
            return 'Room (pending)';
        }
        return null;
    }

    /**
     * Count 'open' orders: those not yet in a terminal status.
     * Terminal statuses chosen: Delivered, Cancelled.
     */
    public static function openOrdersCount(): int
    {
        return static::whereNotIn('status', ['Delivered','Cancelled'])->count();
    }
}
