<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    use HasFactory;

    protected $fillable = ['key','value','type','group','description','autoload'];

    protected $casts = [
        'autoload' => 'boolean',
    ];

    public static function get(string $key, $default = null)
    {
        return Cache::rememberForever("setting:".$key, function () use ($key, $default) {
            $row = static::query()->where('key', $key)->first();
            if (!$row) return $default;
            return static::castFromType($row->value, $row->type);
        });
    }

    public static function set(string $key, $value, string $type = 'string', ?string $group = null, ?string $description = null, bool $autoload = true): void
    {
        $encoded = static::encodeForType($value, $type);
        static::updateOrCreate(
            ['key' => $key],
            compact('encoded','type','group','description','autoload') + ['value' => $encoded]
        );
        Cache::forget("setting:".$key);
        if ($autoload) Cache::forget('settings:autoload');
    }

    public static function allAutoloaded(): array
    {
        return Cache::rememberForever('settings:autoload', function () {
            return static::query()
                ->where('autoload', true)
                ->get()
                ->mapWithKeys(function ($row) {
                    return [
                        $row->key => static::castFromType($row->value, $row->type),
                    ];
                })->toArray();
        });
    }

    protected static function castFromType(?string $value, string $type)
    {
        if ($value === null) return null;
        return match ($type) {
            'int' => (int) $value,
            'bool' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'decimal' => (float) $value,
            'json' => json_decode($value, true),
            default => $value,
        };
    }

    protected static function encodeForType($value, string $type): ?string
    {
        if ($value === null) return null;
        return match ($type) {
            'json' => json_encode($value),
            'bool' => $value ? '1' : '0',
            default => (string) $value,
        };
    }
}
