<?php

namespace App\Services\Delivery;

use App\Services\Geo\GeoServiceInterface;
use Illuminate\Support\Facades\Log;

/**
 * Phase 2 skeleton: Accepts room deliveries automatically; external deliveries are ACCEPT (unverified)
 * until Phase 4 introduces full geocode + radius enforcement + manual review logic.
 */
class DeliveryVerificationService
{
    public function __construct(private GeoServiceInterface $geo) {}

    public function decide(DeliveryContext $ctx): VerificationDecision
    {
        if ($ctx->deliveryMode === 'room') {
            return VerificationDecision::accept('room_delivery', 0.0, $ctx->hotelLat, $ctx->hotelLng, [ 'mode' => 'room' ]);
        }

        // External flow (Phase 2 placeholder): optionally attempt geocode but do not enforce.
        $lat = $lng = $distance = null;
        $raw = $ctx->rawAddress;
        $geoResult = null;
        if ($raw) {
            $geoResult = $this->geo->geocode($raw);
            if ($geoResult) {
                $lat = $geoResult->lat; $lng = $geoResult->lng;
            }
        }
        Log::info('delivery.verification.placeholder', [
            'mode' => 'external',
            'raw' => $raw,
            'geocoded' => (bool)$geoResult,
        ]);
        return VerificationDecision::accept(
            reason: $geoResult ? 'external_placeholder_geocoded' : 'external_placeholder_no_geocode',
            distance: $distance,
            lat: $lat,
            lng: $lng,
            meta: [ 'phase' => 2, 'geocoded' => (bool)$geoResult ]
        );
    }
}
