<?php

namespace App\Services\Geo;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Google geocode adapter (stubbed for now – will be fully implemented in Phase 4).
 * Currently returns null if API key missing or request fails, allowing graceful fallback.
 */
class GoogleGeoService implements GeoServiceInterface
{
    public function __construct(private ?string $apiKey = null) {}

    public function geocode(string $raw): ?GeoResult
    {
        $raw = trim($raw);
        if ($raw === '') return null;
        if (!$this->apiKey) {
            // Defer real implementation until Phase 4
            return null;
        }
        try {
            $resp = Http::timeout(5)->get('https://maps.googleapis.com/maps/api/geocode/json', [
                'address' => $raw,
                'key' => $this->apiKey,
            ]);
            if (!$resp->ok()) return null;
            $data = $resp->json();
            if (($data['status'] ?? '') !== 'OK') return null;
            $first = $data['results'][0] ?? null;
            if (!$first) return null;
            $loc = $first['geometry']['location'] ?? null;
            if (!$loc) return null;
            return new GeoResult(
                raw: $raw,
                formatted: $first['formatted_address'] ?? null,
                lat: (float) $loc['lat'],
                lng: (float) $loc['lng'],
                provider: 'google',
                components: $first['address_components'] ?? []
            );
        } catch (\Throwable $e) {
            Log::warning('GoogleGeoService geocode error', ['message' => $e->getMessage()]);
            return null;
        }
    }
}
