<?php

namespace App\Services\Inventory;

use App\Models\Dish;
use App\Models\InventoryAlert;
use App\Models\InventoryMovement;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ForecastingService
{
    /**
     * Calculate and update velocity for all dishes
     */
    public function updateVelocityForAllDishes(): array
    {
        $results = [
            'processed' => 0,
            'updated' => 0,
            'errors' => 0,
        ];

        Dish::chunk(100, function ($dishes) use (&$results) {
            foreach ($dishes as $dish) {
                try {
                    $this->updateDishVelocity($dish);
                    $results['updated']++;
                } catch (\Exception $e) {
                    Log::error("Failed to update velocity for dish {$dish->id}: " . $e->getMessage());
                    $results['errors']++;
                }
                $results['processed']++;
            }
        });

        Log::info("Velocity update completed", $results);
        return $results;
    }

    /**
     * Calculate and update velocity for a specific dish
     */
    public function updateDishVelocity(Dish $dish): void
    {
        $now = Carbon::now();
        
        // Calculate 7-day velocity
        $velocity7d = $this->calculateVelocity($dish->id, $now->copy()->subDays(7), $now);
        
        // Calculate 30-day velocity
        $velocity30d = $this->calculateVelocity($dish->id, $now->copy()->subDays(30), $now);
        
        // Calculate projected runway
        $projectedRunway = $velocity7d > 0 ? $dish->stock / $velocity7d : 999;
        
        // Update dish with new metrics
        $dish->update([
            'velocity_7d' => round($velocity7d, 2),
            'velocity_30d' => round($velocity30d, 2),
            'projected_runway_days' => round(min($projectedRunway, 999), 1),
            'velocity_updated_at' => $now,
        ]);
    }

    /**
     * Calculate velocity (units per day) for a dish within a date range
     */
    private function calculateVelocity(int $dishId, Carbon $startDate, Carbon $endDate): float
    {
        $totalSold = InventoryMovement::where('dish_id', $dishId)
            ->where('reason', 'sale')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->sum(DB::raw('ABS(`change`)'));

        $days = max(1, $startDate->diffInDays($endDate));
        
        return $totalSold / $days;
    }

    /**
     * Generate and process alerts for all dishes
     */
    public function generateAlerts(): array
    {
        $results = [
            'processed' => 0,
            'alerts_created' => 0,
            'alerts_resolved' => 0,
        ];

        // First, resolve alerts that are no longer valid
        $this->resolveOutdatedAlerts($results);

        // Then generate new alerts
        Dish::with('activeAlerts')->chunk(100, function ($dishes) use (&$results) {
            foreach ($dishes as $dish) {
                $this->processDishAlerts($dish, $results);
                $results['processed']++;
            }
        });

        Log::info("Alert generation completed", $results);
        return $results;
    }

    /**
     * Process alerts for a specific dish
     */
    private function processDishAlerts(Dish $dish, array &$results): void
    {
        $activeAlerts = $dish->activeAlerts->keyBy('type');

        // Check for out of stock
        if ($dish->isOutOfStock()) {
            if (!$activeAlerts->has(InventoryAlert::TYPE_OUT_OF_STOCK)) {
                $this->createAlert($dish, InventoryAlert::TYPE_OUT_OF_STOCK, InventoryAlert::SEVERITY_CRITICAL, [
                    'title' => "OUT OF STOCK: {$dish->name}",
                    'message' => "This dish is completely out of stock and cannot fulfill new orders.",
                ]);
                $results['alerts_created']++;
            }
        }
        // Check for low stock
        elseif ($dish->isLowStock()) {
            if (!$activeAlerts->has(InventoryAlert::TYPE_LOW_STOCK)) {
                $severity = $dish->stock <= ($dish->low_stock_threshold * 0.5) 
                    ? InventoryAlert::SEVERITY_HIGH 
                    : InventoryAlert::SEVERITY_MEDIUM;
                    
                $this->createAlert($dish, InventoryAlert::TYPE_LOW_STOCK, $severity, [
                    'title' => "Low Stock Alert: {$dish->name}",
                    'message' => "Stock level ({$dish->stock}) is below threshold ({$dish->low_stock_threshold}).",
                ]);
                $results['alerts_created']++;
            }
        }

        // Check for safety stock breach
        if ($dish->safety_stock > 0 && $dish->isBelowSafetyStock()) {
            if (!$activeAlerts->has(InventoryAlert::TYPE_SAFETY_STOCK_BREACH)) {
                $this->createAlert($dish, InventoryAlert::TYPE_SAFETY_STOCK_BREACH, InventoryAlert::SEVERITY_HIGH, [
                    'title' => "Safety Stock Breach: {$dish->name}",
                    'message' => "Stock level ({$dish->stock}) is below safety stock ({$dish->safety_stock}).",
                ]);
                $results['alerts_created']++;
            }
        }

        // Check for projected stockout (within 7 days)
        if ($dish->willStockOutWithin(7) && !$dish->isOutOfStock()) {
            if (!$activeAlerts->has(InventoryAlert::TYPE_PROJECTED_STOCKOUT)) {
                $projectedDays = $dish->getProjectedStockoutDays();
                $severity = $projectedDays <= 3 
                    ? InventoryAlert::SEVERITY_HIGH 
                    : InventoryAlert::SEVERITY_MEDIUM;
                    
                $this->createAlert($dish, InventoryAlert::TYPE_PROJECTED_STOCKOUT, $severity, [
                    'title' => "Projected Stockout: {$dish->name}",
                    'message' => "Based on current velocity, this dish will run out in {$projectedDays} days.",
                    'projected_stockout_days' => $projectedDays,
                ]);
                $results['alerts_created']++;
            }
        }
    }

    /**
     * Create a new alert
     */
    private function createAlert(Dish $dish, string $type, string $severity, array $data): InventoryAlert
    {
        return InventoryAlert::create([
            'dish_id' => $dish->id,
            'type' => $type,
            'severity' => $severity,
            'title' => $data['title'],
            'message' => $data['message'],
            'projected_stockout_days' => $data['projected_stockout_days'] ?? null,
            'current_stock' => $dish->stock,
            'current_velocity' => $dish->velocity_7d,
            'metadata' => [
                'dish_name' => $dish->name,
                'velocity_7d' => $dish->velocity_7d,
                'velocity_30d' => $dish->velocity_30d,
                'low_stock_threshold' => $dish->low_stock_threshold,
                'safety_stock' => $dish->safety_stock,
                'reorder_point' => $dish->reorder_point,
                'generated_at' => now()->toISOString(),
            ],
        ]);
    }

    /**
     * Resolve alerts that are no longer valid
     */
    private function resolveOutdatedAlerts(array &$results): void
    {
        // Resolve out of stock alerts where stock is now available
        $resolvedOutOfStock = InventoryAlert::active()
            ->where('type', InventoryAlert::TYPE_OUT_OF_STOCK)
            ->whereHas('dish', function ($query) {
                $query->where('stock', '>', 0);
            })
            ->update(['status' => InventoryAlert::STATUS_RESOLVED]);

        // Resolve low stock alerts where stock is now above threshold
        $resolvedLowStock = InventoryAlert::active()
            ->where('type', InventoryAlert::TYPE_LOW_STOCK)
            ->whereHas('dish', function ($query) {
                $query->whereRaw('stock > low_stock_threshold');
            })
            ->update(['status' => InventoryAlert::STATUS_RESOLVED]);

        // Resolve safety stock alerts where stock is now above safety level
        $resolvedSafetyStock = InventoryAlert::active()
            ->where('type', InventoryAlert::TYPE_SAFETY_STOCK_BREACH)
            ->whereHas('dish', function ($query) {
                $query->whereRaw('stock > safety_stock');
            })
            ->update(['status' => InventoryAlert::STATUS_RESOLVED]);

        // Resolve projected stockout alerts where prediction is no longer valid
        $resolvedProjected = InventoryAlert::active()
            ->where('type', InventoryAlert::TYPE_PROJECTED_STOCKOUT)
            ->whereHas('dish', function ($query) {
                $query->where(function ($q) {
                    $q->where('velocity_7d', '<=', 0)
                      ->orWhereRaw('(stock / velocity_7d) > 7')
                      ->orWhere('stock', '<=', 0);
                });
            })
            ->update(['status' => InventoryAlert::STATUS_RESOLVED]);

        $results['alerts_resolved'] = $resolvedOutOfStock + $resolvedLowStock + $resolvedSafetyStock + $resolvedProjected;
    }

    /**
     * Get critical alerts summary
     */
    public function getCriticalAlertsSummary(): array
    {
        $alerts = InventoryAlert::active()
            ->with('dish:id,name')
            ->orderBy('severity', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();

        return [
            'total_active' => $alerts->count(),
            'critical' => $alerts->where('severity', InventoryAlert::SEVERITY_CRITICAL)->count(),
            'high' => $alerts->where('severity', InventoryAlert::SEVERITY_HIGH)->count(),
            'medium' => $alerts->where('severity', InventoryAlert::SEVERITY_MEDIUM)->count(),
            'low' => $alerts->where('severity', InventoryAlert::SEVERITY_LOW)->count(),
            'by_type' => [
                'out_of_stock' => $alerts->where('type', InventoryAlert::TYPE_OUT_OF_STOCK)->count(),
                'low_stock' => $alerts->where('type', InventoryAlert::TYPE_LOW_STOCK)->count(),
                'projected_stockout' => $alerts->where('type', InventoryAlert::TYPE_PROJECTED_STOCKOUT)->count(),
                'safety_stock_breach' => $alerts->where('type', InventoryAlert::TYPE_SAFETY_STOCK_BREACH)->count(),
            ],
        ];
    }

    /**
     * Get velocity statistics for all dishes
     */
    public function getVelocityStatistics(): array
    {
        $stats = Dish::selectRaw('
            AVG(velocity_7d) as avg_velocity_7d,
            AVG(velocity_30d) as avg_velocity_30d,
            MAX(velocity_7d) as max_velocity_7d,
            MIN(velocity_7d) as min_velocity_7d,
            COUNT(*) as total_dishes,
            COUNT(CASE WHEN velocity_7d > 0 THEN 1 END) as dishes_with_sales
        ')->first();

        return [
            'average_velocity_7d' => round($stats->avg_velocity_7d ?? 0, 2),
            'average_velocity_30d' => round($stats->avg_velocity_30d ?? 0, 2),
            'max_velocity_7d' => round($stats->max_velocity_7d ?? 0, 2),
            'min_velocity_7d' => round($stats->min_velocity_7d ?? 0, 2),
            'total_dishes' => $stats->total_dishes,
            'dishes_with_sales' => $stats->dishes_with_sales,
            'dishes_without_sales' => $stats->total_dishes - $stats->dishes_with_sales,
        ];
    }
}