<?php

namespace App\Services\Inventory;

use App\Models\Dish;
use App\Models\InventoryMovement;
use App\Models\Order;
use App\Models\User;
use App\Models\Location;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

/**
 * InventoryService - Centralized inventory management service
 * 
 * This service provides a single point of control for all inventory movements
 * and stock adjustments, ensuring consistency, auditability, and data integrity.
 * 
 * Key Features:
 * - Pessimistic locking to prevent race conditions
 * - Structured movement reasons with validation
 * - Automatic inventory movement logging
 * - Non-negative stock enforcement
 * - Transaction safety
 */
class InventoryService
{
    /**
     * Canonical movement reasons as defined in Inventory.md
     */
    public const MOVEMENT_REASONS = [
        'sale' => 'Sale/Order fulfillment',
        'adjustment' => 'Manual stock adjustment',
        'correction' => 'Data correction/fix',
        'receipt' => 'Supplier inbound (future)',
        'spoilage' => 'Product spoilage (future)',
        'shrinkage' => 'Inventory shrinkage (future)',
        'return' => 'Customer return (future)',
        'transfer_out' => 'Transfer out (multi-location)',
        'transfer_in' => 'Transfer in (multi-location)',
    ];

    /**
     * Record a sale and decrement stock
     * 
     * @param Dish $dish The dish being sold
     * @param int $quantity Quantity sold
     * @param Order|null $order Associated order
     * @param User|null $user User making the sale (defaults to authenticated user)
     * @param string|null $note Optional note
     * @return InventoryMovement
     * @throws ValidationException If insufficient stock
     */
    public function recordSale(Dish $dish, int $quantity, ?Order $order = null, ?User $user = null, ?string $note = null, ?Location $location = null): InventoryMovement
    {
        if ($quantity <= 0) {
            throw ValidationException::withMessages([
                'quantity' => 'Sale quantity must be positive'
            ]);
        }

        return $this->createMovement(
            dish: $dish,
            change: -$quantity,
            reason: 'sale',
            order: $order,
            user: $user ?? Auth::user(),
            note: $note,
            location: $location
        );
    }

    /**
     * Record a manual stock adjustment
     * 
     * @param Dish $dish The dish being adjusted
     * @param int $change Stock change (positive for increase, negative for decrease)
     * @param string $reason Reason for adjustment
     * @param User|null $user User making the adjustment
     * @param string|null $note Optional note
     * @return InventoryMovement
     * @throws ValidationException If invalid parameters
     */
    public function adjust(Dish $dish, int $change, string $reason = 'adjustment', ?User $user = null, ?string $note = null, ?Location $location = null): InventoryMovement
    {
        if ($change == 0) {
            throw ValidationException::withMessages([
                'change' => 'Stock change cannot be zero'
            ]);
        }

        return $this->createMovement(
            dish: $dish,
            change: $change,
            reason: $reason,
            order: null,
            user: $user ?? Auth::user(),
            note: $note,
            location: $location
        );
    }

    /**
     * Transfer stock between two locations for the same dish.
     * Creates paired movements: transfer_out (negative) and transfer_in (positive), grouped by transfer_group.
     */
    public function transfer(Dish $dish, int $quantity, Location $from, Location $to, ?User $user = null, ?string $note = null): array
    {
        if ($quantity <= 0) {
            throw ValidationException::withMessages([
                'quantity' => 'Transfer quantity must be positive'
            ]);
        }

        $group = 'txf_'.uniqid();

        $out = $this->createMovement(
            dish: $dish,
            change: -$quantity,
            reason: 'transfer_out',
            order: null,
            user: $user ?? Auth::user(),
            note: $note,
            location: $from,
            transferGroup: $group
        );

        $in = $this->createMovement(
            dish: $dish,
            change: $quantity,
            reason: 'transfer_in',
            order: null,
            user: $user ?? Auth::user(),
            note: $note,
            location: $to,
            transferGroup: $group
        );

        return [$out, $in];
    }

    /**
     * Get current available stock for a dish
     * For Phase 1, this is simply the dish.stock value
     * In future phases, this will account for allocations
     * 
     * @param Dish $dish
     * @return int Available stock quantity
     */
    public function getAvailableStock(Dish $dish): int
    {
        return max(0, (int)($dish->stock ?? 0));
    }

    /**
     * Check if sufficient stock is available for a sale
     * 
     * @param Dish $dish
     * @param int $requestedQuantity
     * @return bool
     */
    public function isAvailable(Dish $dish, int $requestedQuantity): bool
    {
        return $this->getAvailableStock($dish) >= $requestedQuantity;
    }

    /**
     * Validate movement reason against canonical list
     * 
     * @param string $reason
     * @return bool
     */
    public function isValidReason(string $reason): bool
    {
        return array_key_exists($reason, self::MOVEMENT_REASONS);
    }

    /**
     * Get list of valid movement reasons
     * 
     * @return array
     */
    public function getValidReasons(): array
    {
        return self::MOVEMENT_REASONS;
    }

    /**
     * Core method to create an inventory movement with proper locking and validation
     * 
     * @param Dish $dish
     * @param int $change Stock change (negative for decrease, positive for increase)
     * @param string $reason Movement reason
     * @param Order|null $order Associated order
     * @param User|null $user User responsible for the movement
     * @param string|null $note Optional note
     * @return InventoryMovement
     * @throws ValidationException If validation fails
     */
    private function createMovement(
        Dish $dish,
        int $change,
        string $reason,
        ?Order $order = null,
        ?User $user = null,
        ?string $note = null,
        ?Location $location = null,
        ?string $transferGroup = null
    ): InventoryMovement {
        // Validate movement reason
        if (!$this->isValidReason($reason)) {
            throw ValidationException::withMessages([
                'reason' => "Invalid movement reason: {$reason}. Valid reasons: " . implode(', ', array_keys(self::MOVEMENT_REASONS))
            ]);
        }

    return DB::transaction(function () use ($dish, $change, $reason, $order, $user, $note, $location, $transferGroup) {
            // Apply pessimistic lock to prevent race conditions
            $dish = Dish::where('id', $dish->id)->lockForUpdate()->first();
            
            if (!$dish) {
                throw ValidationException::withMessages([
                    'dish' => 'Dish not found'
                ]);
            }

            $currentStock = (int)($dish->stock ?? 0);
            $newStock = $currentStock + $change;

            // Enforce non-negative stock rule
            if ($newStock < 0) {
                throw ValidationException::withMessages([
                    'stock' => "Insufficient stock. Available: {$currentStock}, Requested: " . abs($change)
                ]);
            }

            // Update dish stock
            $dish->update(['stock' => $newStock]);

            // Create inventory movement record
            $movement = InventoryMovement::create([
                'dish_id' => $dish->id,
                'location_id' => $location?->id,
                'change' => $change,
                'reason' => $reason,
                'transfer_group' => $transferGroup,
                'order_id' => $order?->id,
                'user_id' => $user?->id,
                'note' => $note,
            ]);

            return $movement;
        });
    }
}