<?php

namespace App\Services\Inventory;

use App\Models\Dish;
use App\Models\InventoryStats;
use App\Models\InventorySnapshot;
use App\Models\InventoryMovement;
use App\Models\InventoryAlert;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PerformanceService
{
    /**
     * Update or create inventory stats for all dishes
     */
    public function updateAllInventoryStats(): array
    {
        $results = [
            'processed' => 0,
            'updated' => 0,
            'created' => 0,
            'errors' => 0,
        ];

        Dish::chunk(100, function ($dishes) use (&$results) {
            foreach ($dishes as $dish) {
                try {
                    $this->updateDishStats($dish);
                    $this->updatePerLocationStats($dish);
                    
                    $existingStats = InventoryStats::where('dish_id', $dish->id)->exists();
                    if ($existingStats) {
                        $results['updated']++;
                    } else {
                        $results['created']++;
                    }
                } catch (\Exception $e) {
                    Log::error("Failed to update stats for dish {$dish->id}: " . $e->getMessage());
                    $results['errors']++;
                }
                $results['processed']++;
            }
        });

        Log::info("Inventory stats update completed", $results);
        return $results;
    }

    /**
     * Update inventory stats for a specific dish
     */
    public function updateDishStats(Dish $dish): InventoryStats
    {
        $now = Carbon::now();
        
        // Calculate movement aggregates
        $movements = InventoryMovement::where('dish_id', $dish->id);
        
        $totalSales = $movements->clone()->where('reason', 'sale')->sum(DB::raw('ABS(`change`)'));
        $totalAdjustments = $movements->clone()->where('reason', 'adjustment')->sum(DB::raw('ABS(`change`)'));
        $totalMovements = $movements->clone()->count();
        
        // Get sales value (approximate from movements and current price)
        $totalSalesValue = $totalSales * $dish->price;
        $averageSalePrice = $totalSales > 0 ? $totalSalesValue / $totalSales : $dish->price;
        
        // Calculate inventory value (cost basis approximation)
        $inventoryValue = $dish->stock * ($dish->price * 0.7); // Assume 30% margin
        
        // Get last movement timestamps
        $lastMovement = $movements->clone()->latest()->first();
        $lastSale = $movements->clone()->where('reason', 'sale')->latest()->first();
        
        // Count events
        $stockoutEvents = $this->countStockoutEvents($dish->id);
        $lowStockEvents = $this->countLowStockEvents($dish->id);
        $reorderEvents = $this->countReorderEvents($dish->id);
        
        // Get last stockout timestamp
        $lastStockout = InventoryAlert::where('dish_id', $dish->id)
            ->where('type', InventoryAlert::TYPE_OUT_OF_STOCK)
            ->latest()
            ->first();

        return InventoryStats::updateOrCreate(
            ['dish_id' => $dish->id, 'location_id' => null],
            [
                'on_hand' => $dish->stock,
                'allocated' => 0, // Allocation not implemented yet
                'available' => max(0, $dish->stock), // on_hand - allocated
                'total_sales' => $totalSales,
                'total_adjustments' => $totalAdjustments,
                'total_movements' => $totalMovements,
                'velocity_7d' => $dish->velocity_7d ?? 0,
                'velocity_30d' => $dish->velocity_30d ?? 0,
                'projected_runway_days' => $dish->projected_runway_days,
                'total_sales_value' => $totalSalesValue,
                'average_sale_price' => $averageSalePrice,
                'inventory_value' => $inventoryValue,
                'stockout_events' => $stockoutEvents,
                'low_stock_events' => $lowStockEvents,
                'reorder_events' => $reorderEvents,
                'last_movement_at' => $lastMovement?->created_at,
                'last_sale_at' => $lastSale?->created_at,
                'last_stockout_at' => $lastStockout?->created_at,
                'stats_calculated_at' => $now,
            ]
        );
    }

    /**
     * Update per-location stats for a dish across all active locations
     */
    private function updatePerLocationStats(Dish $dish): void
    {
        // Aggregate movements by location for this dish
        $byLocation = InventoryMovement::select(
                'location_id',
                DB::raw('COALESCE(SUM(`change`),0) as qty')
            )
            ->where('dish_id', $dish->id)
            ->whereNotNull('location_id')
            ->groupBy('location_id')
            ->get();

        // Ensure we cover all existing locations even if qty is zero (optional)
        $locations = \App\Models\Location::where('is_active', true)->get(['id']);
        $map = $byLocation->keyBy('location_id');

        foreach ($locations as $loc) {
            $locId = $loc->id;
            $qty = (int) ($map[$locId]->qty ?? 0);

            InventoryStats::updateOrCreate(
                ['dish_id' => $dish->id, 'location_id' => $locId],
                [
                    'on_hand' => $qty,
                    'allocated' => 0,
                    'available' => max(0, $qty),
                    // Per-location aggregates: compute movements scoped to dish+location
                    'total_sales' => InventoryMovement::where('dish_id', $dish->id)->where('location_id', $locId)->where('reason', 'sale')->sum(DB::raw('ABS(`change`)')),
                    'total_adjustments' => InventoryMovement::where('dish_id', $dish->id)->where('location_id', $locId)->where('reason', 'adjustment')->sum(DB::raw('ABS(`change`)')),
                    'total_movements' => InventoryMovement::where('dish_id', $dish->id)->where('location_id', $locId)->count(),
                    'velocity_7d' => $dish->velocity_7d ?? 0,
                    'velocity_30d' => $dish->velocity_30d ?? 0,
                    'projected_runway_days' => null,
                    'total_sales_value' => ($dish->price) * (InventoryMovement::where('dish_id', $dish->id)->where('location_id', $locId)->where('reason', 'sale')->sum(DB::raw('ABS(`change`)'))),
                    'average_sale_price' => $dish->price,
                    'inventory_value' => $qty * ($dish->price * 0.7),
                    'stockout_events' => 0,
                    'low_stock_events' => 0,
                    'reorder_events' => 0,
                    'last_movement_at' => InventoryMovement::where('dish_id', $dish->id)->where('location_id', $locId)->latest()->value('created_at'),
                    'last_sale_at' => InventoryMovement::where('dish_id', $dish->id)->where('location_id', $locId)->where('reason', 'sale')->latest()->value('created_at'),
                    'last_stockout_at' => null,
                    'stats_calculated_at' => Carbon::now(),
                ]
            );
        }
    }

    /**
     * Create daily snapshots for all dishes
     */
    public function createDailySnapshots(Carbon $date = null): array
    {
        $date = $date ?? Carbon::yesterday();
        $results = [
            'date' => $date->format('Y-m-d'),
            'dish_snapshots' => 0,
            'global_snapshot' => false,
            'errors' => 0,
        ];

        // Check if snapshots already exist for this date
        if (InventorySnapshot::where('snapshot_date', $date->format('Y-m-d'))->exists()) {
            Log::info("Snapshots already exist for {$date->format('Y-m-d')}");
            return $results;
        }

        // Create dish-specific snapshots
        Dish::chunk(100, function ($dishes) use ($date, &$results) {
            foreach ($dishes as $dish) {
                try {
                    $this->createDishSnapshot($dish, $date);
                    $results['dish_snapshots']++;
                } catch (\Exception $e) {
                    Log::error("Failed to create snapshot for dish {$dish->id}: " . $e->getMessage());
                    $results['errors']++;
                }
            }
        });

        // Create global snapshot
        try {
            $this->createGlobalSnapshot($date);
            $results['global_snapshot'] = true;
        } catch (\Exception $e) {
            Log::error("Failed to create global snapshot: " . $e->getMessage());
            $results['errors']++;
        }

        Log::info("Daily snapshots created", $results);
        return $results;
    }

    /**
     * Create snapshot for a specific dish
     */
    private function createDishSnapshot(Dish $dish, Carbon $date): InventorySnapshot
    {
        // Calculate daily activity for the given date
        $dailyMovements = InventoryMovement::where('dish_id', $dish->id)
            ->whereDate('created_at', $date)
            ->get();

        $salesQty = $dailyMovements->where('reason', 'sale')->sum(function ($movement) {
            return abs($movement->change);
        });

        $adjustmentsQty = $dailyMovements->where('reason', 'adjustment')->sum(function ($movement) {
            return abs($movement->change);
        });

        $salesValue = $salesQty * $dish->price;

        // Check alert status
        $alertsQuery = InventoryAlert::where('dish_id', $dish->id)
            ->whereDate('created_at', '<=', $date);

        $hadStockoutAlert = $alertsQuery->clone()
            ->where('type', InventoryAlert::TYPE_OUT_OF_STOCK)
            ->exists();

        $activeAlertsCount = $alertsQuery->clone()
            ->where('status', InventoryAlert::STATUS_ACTIVE)
            ->count();

        // Determine stock status during the day
        $wasOutOfStock = $dish->stock <= 0 || $hadStockoutAlert;
        $wasLowStock = $dish->isLowStock() || 
            $alertsQuery->clone()->where('type', InventoryAlert::TYPE_LOW_STOCK)->exists();

        return InventorySnapshot::create([
            'snapshot_date' => $date->format('Y-m-d'),
            'dish_id' => $dish->id,
            'type' => InventorySnapshot::TYPE_DISH,
            'on_hand' => $dish->stock,
            'allocated' => 0, // Allocation not implemented yet
            'available' => max(0, $dish->stock),
            'sales_qty' => $salesQty,
            'sales_value' => $salesValue,
            'adjustments_qty' => $adjustmentsQty,
            'movements_count' => $dailyMovements->count(),
            'velocity_7d' => $dish->velocity_7d ?? 0,
            'velocity_30d' => $dish->velocity_30d ?? 0,
            'projected_runway_days' => $dish->projected_runway_days,
            'inventory_value' => $dish->stock * ($dish->price * 0.7), // Cost approximation
            'average_sale_price' => $dish->price,
            'was_out_of_stock' => $wasOutOfStock,
            'was_low_stock' => $wasLowStock,
            'had_stockout_alert' => $hadStockoutAlert,
            'active_alerts_count' => $activeAlertsCount,
            'fill_rate' => $salesQty > 0 ? 100.0 : null, // Simplified fill rate
            'stockout_events' => $wasOutOfStock ? 1 : 0,
        ]);
    }

    /**
     * Create global snapshot for the date
     */
    private function createGlobalSnapshot(Carbon $date): InventorySnapshot
    {
        $dishes = Dish::all();
        
        $totalDishes = $dishes->count();
        $dishesWithStock = $dishes->where('stock', '>', 0)->count();
        $dishesLowStock = $dishes->filter(fn($dish) => $dish->isLowStock())->count();
        $dishesOutOfStock = $dishes->where('stock', '<=', 0)->count();
        
        $totalInventoryValue = $dishes->sum(function ($dish) {
            return $dish->stock * ($dish->price * 0.7); // Cost approximation
        });

        $avgRunwayDays = $dishes->whereNotNull('projected_runway_days')
            ->avg('projected_runway_days');

        // Calculate daily totals
        $dailyMovements = InventoryMovement::whereDate('created_at', $date)->get();
        $totalSalesQty = $dailyMovements->where('reason', 'sale')->sum(function ($movement) {
            return abs($movement->change);
        });

        $totalSalesValue = $dishes->sum(function ($dish) use ($dailyMovements) {
            $dishSales = $dailyMovements->where('dish_id', $dish->id)
                ->where('reason', 'sale')
                ->sum(function ($movement) {
                    return abs($movement->change);
                });
            return $dishSales * $dish->price;
        });

        return InventorySnapshot::create([
            'snapshot_date' => $date->format('Y-m-d'),
            'dish_id' => null,
            'type' => InventorySnapshot::TYPE_GLOBAL,
            'sales_qty' => $totalSalesQty,
            'sales_value' => $totalSalesValue,
            'movements_count' => $dailyMovements->count(),
            'total_dishes' => $totalDishes,
            'dishes_with_stock' => $dishesWithStock,
            'dishes_low_stock' => $dishesLowStock,
            'dishes_out_of_stock' => $dishesOutOfStock,
            'total_inventory_value' => $totalInventoryValue,
            'avg_runway_days' => $avgRunwayDays,
        ]);
    }

    /**
     * Reconcile stats with ledger to detect drift
     */
    public function reconcileStats(): array
    {
        $results = [
            'dishes_checked' => 0,
            'discrepancies' => 0,
            'corrected' => 0,
            'errors' => [],
        ];

        InventoryStats::with('dish')->chunk(100, function ($stats) use (&$results) {
            foreach ($stats as $stat) {
                try {
                    $this->reconcileDishStats($stat, $results);
                    $results['dishes_checked']++;
                } catch (\Exception $e) {
                    $results['errors'][] = "Dish {$stat->dish_id}: " . $e->getMessage();
                }
            }
        });

        Log::info("Stats reconciliation completed", $results);
        return $results;
    }

    /**
     * Reconcile stats for a specific dish
     */
    private function reconcileDishStats(InventoryStats $stats, array &$results): void
    {
        $dish = $stats->dish;
        
        // Check on_hand discrepancy
        if ($stats->on_hand !== $dish->stock) {
            $results['discrepancies']++;
            
            Log::warning("Stock discrepancy for dish {$dish->id}: stats={$stats->on_hand}, actual={$dish->stock}");
            
            // Update stats to match actual
            $stats->update([
                'on_hand' => $dish->stock,
                'available' => max(0, $dish->stock),
                'stats_calculated_at' => now(),
            ]);
            
            $results['corrected']++;
        }

        // Check movement count discrepancy
        $actualMovements = InventoryMovement::where('dish_id', $dish->id)->count();
        if ($stats->total_movements !== $actualMovements) {
            $results['discrepancies']++;
            
            Log::warning("Movement count discrepancy for dish {$dish->id}: stats={$stats->total_movements}, actual={$actualMovements}");
            
            // Trigger full stats recalculation
            $this->updateDishStats($dish);
            $results['corrected']++;
        }
    }

    /**
     * Get performance benchmarks
     */
    public function getPerformanceBenchmarks(): array
    {
        $start = microtime(true);

        // Benchmark common queries
        $benchmarks = [];

        // Query 1: Get all available dishes
        $queryStart = microtime(true);
        $availableDishes = InventoryStats::available()->count();
        $benchmarks['available_dishes_query'] = [
            'description' => 'Count available dishes',
            'time_ms' => round((microtime(true) - $queryStart) * 1000, 2),
            'result_count' => $availableDishes,
        ];

        // Query 2: Get low stock items
        $queryStart = microtime(true);
        $lowStockItems = InventoryStats::lowStock()->with('dish:id,name')->limit(10)->get();
        $benchmarks['low_stock_query'] = [
            'description' => 'Get low stock items with dish names',
            'time_ms' => round((microtime(true) - $queryStart) * 1000, 2),
            'result_count' => $lowStockItems->count(),
        ];

        // Query 3: Get velocity statistics
        $queryStart = microtime(true);
        $velocityStats = InventoryStats::selectRaw('
            AVG(velocity_7d) as avg_velocity,
            MAX(velocity_7d) as max_velocity,
            COUNT(*) as total_items
        ')->first();
        $benchmarks['velocity_stats_query'] = [
            'description' => 'Calculate velocity statistics',
            'time_ms' => round((microtime(true) - $queryStart) * 1000, 2),
            'avg_velocity' => round($velocityStats->avg_velocity ?? 0, 2),
        ];

        // Query 4: Recent snapshots aggregation
        $queryStart = microtime(true);
        $recentSnapshots = InventorySnapshot::recent(7)
            ->dishSnapshots()
            ->selectRaw('COUNT(*) as total, SUM(sales_qty) as total_sales')
            ->first();
        $benchmarks['recent_snapshots_query'] = [
            'description' => 'Aggregate recent snapshots',
            'time_ms' => round((microtime(true) - $queryStart) * 1000, 2),
            'total_snapshots' => $recentSnapshots->total ?? 0,
        ];

        $benchmarks['total_benchmark_time'] = round((microtime(true) - $start) * 1000, 2);

        return $benchmarks;
    }

    /**
     * Count stockout events for a dish
     */
    private function countStockoutEvents(int $dishId): int
    {
        return InventoryAlert::where('dish_id', $dishId)
            ->where('type', InventoryAlert::TYPE_OUT_OF_STOCK)
            ->count();
    }

    /**
     * Count low stock events for a dish
     */
    private function countLowStockEvents(int $dishId): int
    {
        return InventoryAlert::where('dish_id', $dishId)
            ->where('type', InventoryAlert::TYPE_LOW_STOCK)
            ->count();
    }

    /**
     * Count reorder events for a dish
     */
    private function countReorderEvents(int $dishId): int
    {
        // For now, approximate from safety stock breach alerts
        return InventoryAlert::where('dish_id', $dishId)
            ->where('type', InventoryAlert::TYPE_SAFETY_STOCK_BREACH)
            ->count();
    }
}