<?php

namespace App\Services\Reporting;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\InventoryMovement;

class ReportingService
{
    /**
     * Reporting time windows configuration
     */
    public const WINDOWS = [
        'daily' => [
            'period' => '1 day',
            'cache_ttl' => 3600, // 1 hour
        ],
        'weekly' => [
            'period' => '7 days',
            'cache_ttl' => 7200, // 2 hours
        ],
        'monthly' => [
            'period' => '30 days',
            'cache_ttl' => 14400, // 4 hours
        ],
    ];

    /**
     * Generate comprehensive KPI report for a given period
     */
    public function generateKPIReport(string $window = 'daily', ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        $cacheKey = "kpi_report_{$window}_" . ($startDate?->toDateString() ?? 'current') . '_' . ($endDate?->toDateString() ?? 'current');
        
        return Cache::remember($cacheKey, self::WINDOWS[$window]['cache_ttl'] ?? 3600, function () use ($window, $startDate, $endDate) {
            $dateRange = $this->getDateRange($window, $startDate, $endDate);
            
            return [
                'period' => [
                    'window' => $window,
                    'start_date' => $dateRange['start']->toDateString(),
                    'end_date' => $dateRange['end']->toDateString(),
                ],
                'sales_metrics' => $this->calculateSalesMetrics($dateRange['start'], $dateRange['end']),
                'inventory_metrics' => $this->calculateInventoryMetrics($dateRange['start'], $dateRange['end']),
                'operational_metrics' => $this->calculateOperationalMetrics($dateRange['start'], $dateRange['end']),
                'generated_at' => now()->toISOString(),
            ];
        });
    }

    /**
     * Calculate sales-related metrics
     */
    private function calculateSalesMetrics(Carbon $startDate, Carbon $endDate): array
    {
        // Get completed orders in the period
        $orders = Order::where('status', 'Delivered')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->with('items')
            ->get();

        $grossSales = 0;
        $discounts = 0;
        $refunds = 0; // TODO: Implement refund tracking
        $totalOrders = $orders->count();
        $totalItems = 0;

        foreach ($orders as $order) {
            // Gross Sales = Sum(order_items.price * qty) before discounts
            foreach ($order->items as $item) {
                $grossSales += $item->price * $item->quantity;
                $totalItems += $item->quantity;
            }
            
            // Add order-level discounts
            $discounts += $order->discount ?? 0;
        }

        $netSales = $grossSales - $discounts - $refunds;
        $averageOrderValue = $totalOrders > 0 ? $netSales / $totalOrders : 0;

        return [
            'gross_sales' => round($grossSales, 2),
            'net_sales' => round($netSales, 2),
            'discounts' => round($discounts, 2),
            'refunds' => round($refunds, 2),
            'total_orders' => $totalOrders,
            'total_items_sold' => $totalItems,
            'average_order_value' => round($averageOrderValue, 2),
            'sales_velocity_per_day' => round($totalItems / max(1, $startDate->diffInDays($endDate)), 2),
        ];
    }

    /**
     * Calculate inventory-related metrics
     */
    private function calculateInventoryMetrics(Carbon $startDate, Carbon $endDate): array
    {
        // Current inventory value
        $currentInventoryValue = DB::table('dishes')
            ->whereNull('deleted_at')
            ->selectRaw('SUM(stock * price) as total_value')
            ->value('total_value') ?? 0;

        // Total stock on hand
        $totalOnHand = DB::table('dishes')
            ->whereNull('deleted_at')
            ->sum('stock');

        // Low stock items
        $lowStockCount = DB::table('dishes')
            ->whereNull('deleted_at')
            ->whereRaw('stock <= low_stock_threshold')
            ->count();

        // Out of stock items
        $outOfStockCount = DB::table('dishes')
            ->whereNull('deleted_at')
            ->where('stock', '<=', 0)
            ->count();

        // Sales movements in period for COGS calculation
        $salesMovements = InventoryMovement::where('reason', 'sale')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->with('dish')
            ->get();

        $cogs = 0;
        foreach ($salesMovements as $movement) {
            // For now, use current dish price as cost basis
            // TODO: Implement proper cost tracking when BOM is available
            $unitCost = $movement->dish ? $movement->dish->price * 0.4 : 0; // Assume 40% cost ratio
            $cogs += abs($movement->change) * $unitCost;
        }

        // Calculate inventory turnover
        $averageInventoryValue = $currentInventoryValue; // Simplified - could use period average
        $inventoryTurnover = $averageInventoryValue > 0 ? $cogs / $averageInventoryValue : 0;

        // Days of inventory (runway)
        $totalSold = $salesMovements->sum(function ($movement) {
            return abs($movement->change);
        });
        $dailyVelocity = $totalSold / max(1, $startDate->diffInDays($endDate));
        $daysOfInventory = $dailyVelocity > 0 ? $totalOnHand / $dailyVelocity : 999;

        return [
            'current_inventory_value' => round($currentInventoryValue, 2),
            'total_on_hand' => $totalOnHand,
            'low_stock_count' => $lowStockCount,
            'out_of_stock_count' => $outOfStockCount,
            'cogs' => round($cogs, 2),
            'inventory_turnover' => round($inventoryTurnover, 2),
            'days_of_inventory' => round(min($daysOfInventory, 999), 1),
        ];
    }

    /**
     * Calculate operational metrics
     */
    private function calculateOperationalMetrics(Carbon $startDate, Carbon $endDate): array
    {
        $totalOrders = Order::whereBetween('created_at', [$startDate, $endDate])->count();
        $deliveredOrders = Order::where('status', 'Delivered')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();
        $cancelledOrders = Order::where('status', 'Cancelled')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        $fillRate = $totalOrders > 0 ? ($deliveredOrders / $totalOrders) * 100 : 0;
        $cancellationRate = $totalOrders > 0 ? ($cancelledOrders / $totalOrders) * 100 : 0;

        // Stockout events (orders that couldn't be fulfilled due to stock)
        // TODO: Implement proper stockout tracking
        $stockoutEvents = 0;
        $stockoutRate = $totalOrders > 0 ? ($stockoutEvents / $totalOrders) * 100 : 0;

        return [
            'total_orders' => $totalOrders,
            'delivered_orders' => $deliveredOrders,
            'cancelled_orders' => $cancelledOrders,
            'fill_rate_percent' => round($fillRate, 2),
            'cancellation_rate_percent' => round($cancellationRate, 2),
            'stockout_events' => $stockoutEvents,
            'stockout_rate_percent' => round($stockoutRate, 2),
        ];
    }

    /**
     * Get top selling dishes for a period
     */
    public function getTopSellingDishes(string $window = 'daily', int $limit = 10, ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        $dateRange = $this->getDateRange($window, $startDate, $endDate);
        
        return InventoryMovement::where('reason', 'sale')
            ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->select('dish_id', DB::raw('SUM(ABS(`change`)) as total_sold'))
            ->with('dish:id,name,price')
            ->groupBy('dish_id')
            ->orderByDesc('total_sold')
            ->limit($limit)
            ->get()
            ->map(function ($movement) {
                return [
                    'dish_id' => $movement->dish_id,
                    'dish_name' => $movement->dish->name ?? 'Unknown',
                    'total_sold' => $movement->total_sold,
                    'revenue' => round($movement->total_sold * ($movement->dish->price ?? 0), 2),
                ];
            })
            ->toArray();
    }

    /**
     * Calculate gross margin percentage
     */
    public function calculateGrossMargin(float $netSales, float $cogs): float
    {
        return $netSales > 0 ? (($netSales - $cogs) / $netSales) * 100 : 0;
    }

    /**
     * Get date range for a given window
     */
    private function getDateRange(string $window, ?Carbon $startDate = null, ?Carbon $endDate = null): array
    {
        if ($startDate && $endDate) {
            return ['start' => $startDate, 'end' => $endDate];
        }

        $end = now();
        
        switch ($window) {
            case 'daily':
                $start = $end->copy()->startOfDay();
                break;
            case 'weekly':
                $start = $end->copy()->subDays(7)->startOfDay();
                break;
            case 'monthly':
                $start = $end->copy()->subDays(30)->startOfDay();
                break;
            default:
                $start = $end->copy()->startOfDay();
        }

        return ['start' => $start, 'end' => $end];
    }

    /**
     * Clear reporting cache
     */
    public function clearCache(): void
    {
        Cache::flush(); // TODO: More targeted cache clearing
    }
}