<?php

namespace App\Support;

/**
 * Lazy loader + accessor for the generated image variants manifest.
 * Manifest structure (public/assets/image-manifest.json):
 * {
 *   generatedAt: string,
 *   slider: [ { type:'slider', base:'hero', lqip:'assets/slider/hero-lqip.jpg', variants:[ { width:480, webp:'..', avif:'..', jpg:'..' }, ... ] } ],
 *   dishes: [ { type:'dish', base:'jollof', thumb:'assets/dishes/jollof-thumb-200.jpg', lqip:'...', variants:[ ... ] } ]
 * }
 */
class ImageManifest
{
    protected static ?array $manifest = null;
    protected static ?int $loadedAt = null;
    protected static int $ttlSeconds = 30; // allow light auto-reload for dev changes

    public static function path(): string
    {
        // Public path is base_path('public') in Laravel; we avoid helper to keep this class portable.
        return base_path('public/assets/image-manifest.json');
    }

    public static function get(bool $force = false): ?array
    {
        if ($force) {
            static::$manifest = null;
        }
        $now = time();
        if (static::$manifest !== null && static::$loadedAt !== null && ($now - static::$loadedAt) < static::$ttlSeconds) {
            return static::$manifest;
        }
        $file = static::path();
        if (!is_file($file)) {
            static::$manifest = null;
            static::$loadedAt = $now;
            return null;
        }
        try {
            $json = file_get_contents($file);
            $data = json_decode($json, true, 512, JSON_THROW_ON_ERROR);
            if (!is_array($data)) {
                static::$manifest = null;
            } else {
                static::$manifest = $data;
            }
        } catch (\Throwable $e) {
            static::$manifest = null; // fail silent; component will degrade gracefully
        }
        static::$loadedAt = $now;
        return static::$manifest;
    }

    /** Find an entry by type (slider|dish) and base name. */
    public static function find(string $type, string $base): ?array
    {
        $manifest = static::get();
        if (!$manifest) return null;
        $list = $manifest[$type === 'slider' ? 'slider' : 'dishes'] ?? [];
        foreach ($list as $entry) {
            if (($entry['base'] ?? null) === $base) return $entry;
        }
        return null;
    }
}
