<?php

namespace App\Support;

class ImageVariantGenerator
{
    /**
     * Generate WebP variants for a given public path image.
     * Backwards-compatible helper around generateVariants().
     * @param string $publicPath Relative to public/ or absolute filesystem path under public dir
     * @param array<int> $widths
     * @return array{generated:int, skipped:int}
     */
    public static function generateWebpVariants(string $publicPath, array $widths = [320,640,1024,1536]): array
    {
        return static::generateVariants($publicPath, $widths, ['webp']);
    }

    /**
     * Generate variants for supplied formats (webp, avif) using GD if available, else Imagick.
     * @param string $publicPath
     * @param array<int> $widths
     * @param array<string> $formats e.g., ['webp','avif']
     * @return array{generated:int, skipped:int}
     */
    public static function generateVariants(string $publicPath, array $widths = [320,640,1024,1536], array $formats = ['webp','avif']): array
    {
        $root = public_path();
        $srcPath = str_starts_with($publicPath, $root) ? $publicPath : ($root . DIRECTORY_SEPARATOR . ltrim($publicPath, '\/'));
        if (!is_file($srcPath)) return ['generated' => 0, 'skipped' => 0];

        $ext = strtolower(pathinfo($srcPath, PATHINFO_EXTENSION));
        if (!in_array($ext, ['jpg','jpeg','png'], true)) return ['generated' => 0, 'skipped' => 0];

        [$origW, $origH] = @getimagesize($srcPath) ?: [null, null];
        if (!$origW || !$origH) return ['generated' => 0, 'skipped' => 0];

        $generated = 0; $skipped = 0;
        foreach ($widths as $w) {
            if ($w >= $origW) { $skipped++; continue; }
            foreach ($formats as $fmt) {
                $ok = static::resizeAndSave($srcPath, $ext, $origW, $origH, (int)$w, $fmt);
                if ($ok === true) $generated++; elseif ($ok === null) $skipped++; // null means up-to-date or unsupported
            }
        }
        return ['generated' => $generated, 'skipped' => $skipped];
    }

    /**
     * @return bool|null true if written, null if skipped/unsupported
     */
    protected static function resizeAndSave(string $srcPath, string $srcExt, int $origW, int $origH, int $w, string $format): ?bool
    {
        $base = substr($srcPath, 0, - (strlen($srcExt) + 1));
        $dst = $base . '-' . $w . '.' . $format;
        if (is_file($dst) && filemtime($dst) >= filemtime($srcPath)) {
            return null; // up-to-date
        }

        // Try GD first
        $gdWritten = static::resizeWithGd($srcPath, $srcExt, $origW, $origH, $w, $format);
        if ($gdWritten !== false) return $gdWritten ? true : null;

        // Fallback: Imagick
        $imWritten = static::resizeWithImagick($srcPath, $origW, $origH, $w, $format);
        if ($imWritten !== false) return $imWritten ? true : null;

        return null;
    }

    protected static function resizeWithGd(string $srcPath, string $srcExt, int $origW, int $origH, int $w, string $format): bool|int
    {
        $supports = [
            'webp' => function_exists('imagewebp'),
            'avif' => function_exists('imageavif'),
        ];
        if (!($supports[$format] ?? false)) return false; // unsupported by GD

        $load = match($srcExt){
            'jpg','jpeg' => 'imagecreatefromjpeg',
            'png' => 'imagecreatefrompng',
            default => null,
        };
        if (!$load || !function_exists($load)) return false;

        $ratio = $origH / $origW;
        $h = (int) round($w * $ratio);
        $dst = substr($srcPath, 0, - (strlen($srcExt) + 1)) . '-' . $w . '.' . $format;

        $srcImg = @$load($srcPath);
        if (!$srcImg) return false;
        $dstImg = imagecreatetruecolor($w, $h);
        imagealphablending($dstImg, true);
        imagesavealpha($dstImg, true);
        imagecopyresampled($dstImg, $srcImg, 0, 0, 0, 0, $w, $h, $origW, $origH);

        $ok = match($format){
            'webp' => @imagewebp($dstImg, $dst, 82),
            'avif' => @imageavif($dstImg, $dst, 45),
            default => false,
        };
        imagedestroy($dstImg);
        imagedestroy($srcImg);
        return $ok ? 1 : 0;
    }

    /**
     * @suppress PhanUndeclaredClassMethod, PhanUndeclaredClass
     */
    protected static function resizeWithImagick(string $srcPath, int $origW, int $origH, int $w, string $format): bool|int
    {
        if (!class_exists('Imagick')) return false; // extension not installed
        try {
            $h = (int) round($w * ($origH / $origW));
            $dst = substr($srcPath, 0, strrpos($srcPath, '.')) . '-' . $w . '.' . $format;
            // Use dynamic class & constant resolution to avoid static analyzer errors
            $class = 'Imagick';
            /** @var object $img */
            $img = new $class($srcPath);
            $filter = defined($class.'::FILTER_LANCZOS') ? constant($class.'::FILTER_LANCZOS') : 1;
            $img->resizeImage($w, $h, $filter, 1);
            $img->setImageFormat($format);
            if ($format === 'webp') $img->setImageCompressionQuality(82);
            if ($format === 'avif') $img->setOption('heic:speed', '6');
            $ok = $img->writeImage($dst);
            $img->clear();
            $img->destroy();
            return $ok ? 1 : 0;
        } catch (\Throwable $e) {
            return 0;
        }
    }
}
