<?php

namespace App\Support\Images;

use Illuminate\Support\Facades\Storage;

class DishImageVariants
{
    protected array $widths = [200,400,600,800,1200,1600];

    /**
     * Generate resized JPEG variants and a tiny LQIP for a public storage image
     * and merge into public/assets/image-manifest.json.
     *
     * @param string $relativePath e.g. 'dishes/abc123.jpg' relative to storage/public
     */
    public function generate(string $relativePath): void
    {
        $relativePath = ltrim($relativePath, '/');
        $disk = Storage::disk('public');
        if (!$disk->exists($relativePath)) return;

        $sourceFull = $disk->path($relativePath);
        $info = @getimagesize($sourceFull);
        if (!$info) return;
        [$origW,$origH] = [$info[0], $info[1]];
        $mime = $info['mime'] ?? '';

        $create = match($mime) {
            'image/jpeg' => 'imagecreatefromjpeg',
            'image/png'  => 'imagecreatefrompng',
            'image/webp' => (function_exists('imagecreatefromwebp') ? 'imagecreatefromwebp' : null),
            default => null,
        };
        if (!$create || !is_callable($create)) return;
        $src = @($create)($sourceFull);
        if (!$src) return;

        $outDir = public_path('assets/dishes');
        if (!is_dir($outDir)) @mkdir($outDir, 0775, true);

        $baseName = pathinfo($sourceFull, PATHINFO_FILENAME);
        $variants = [];
        foreach ($this->widths as $w) {
            if ($w >= $origW) continue; // do not upscale
            $h = (int) round($w * ($origH / max(1,$origW)));
            $canvas = imagecreatetruecolor($w, $h);
            imagecopyresampled($canvas, $src, 0,0,0,0, $w,$h, $origW,$origH);
            $jpgRel = 'assets/dishes/'.$baseName.'-'.$w.'.jpg';
            imagejpeg($canvas, public_path($jpgRel), 80);
            imagedestroy($canvas);
            $variants[] = [ 'width' => $w, 'jpg' => $jpgRel ];
        }

        // LQIP
        $lqipRel = null;
        if ($origW > 0) {
            $lw = 24; $lh = (int) round($lw * ($origH / max(1,$origW)));
            $lq = imagecreatetruecolor($lw, $lh);
            imagecopyresampled($lq, $src, 0,0,0,0, $lw,$lh, $origW,$origH);
            $lqipRel = 'assets/dishes/'.$baseName.'-lqip.jpg';
            imagejpeg($lq, public_path($lqipRel), 35);
            imagedestroy($lq);
        }

        imagedestroy($src);

        // Update manifest
        $manifestFile = public_path('assets/image-manifest.json');
        $manifest = [ 'generatedAt' => now()->toIso8601String(), 'slider' => [], 'dishes' => [] ];
        if (is_file($manifestFile)) {
            try {
                $json = file_get_contents($manifestFile);
                $dec = json_decode($json, true);
                if (is_array($dec)) $manifest = array_merge($manifest, $dec);
            } catch (\Throwable $e) {}
        }
        $manifest['dishes'] = array_values(array_filter($manifest['dishes'], fn($e)=> ($e['base']??null) !== $baseName));
        $manifest['dishes'][] = [
            'type' => 'dish',
            'base' => $baseName,
            'variants' => $variants,
            'thumb' => null,
            'lqip' => $lqipRel,
            'original' => 'storage/dishes/'.basename($relativePath),
            'partial' => true,
        ];
        $manifest['generatedAt'] = now()->toIso8601String();
        @file_put_contents($manifestFile, json_encode($manifest, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
    }
}
