<?php

namespace App\Support\Orders;

use App\Enums\OrderState;
use App\Models\Order;
use App\Events\OrderStateChanged;

class OrderLifecycle
{
    /**
     * Transition map (enum-based) mirrors model static map but centralized here for future expansion (events, guards).
     */
    public const TRANSITIONS = [
        'awaiting_payment' => [OrderState::AWAITING_REVIEW, OrderState::AWAITING_KITCHEN, OrderState::CANCELLED],
        'awaiting_review'  => [OrderState::AWAITING_KITCHEN, OrderState::CANCELLED],
        'awaiting_kitchen' => [OrderState::PREPARING, OrderState::CANCELLED],
        'preparing'        => [OrderState::READY, OrderState::CANCELLED],
        'ready'            => [OrderState::DELIVERING, OrderState::COMPLETED, OrderState::CANCELLED],
        'delivering'       => [OrderState::COMPLETED, OrderState::CANCELLED],
        'completed'        => [],
        'cancelled'        => [],
    ];

    public function can(Order $order, OrderState $to): bool
    {
    $current = $this->currentState($order);
    return in_array($to, self::TRANSITIONS[$current->value] ?? [], true);
    }

    public function transition(Order $order, OrderState $to, bool $force = false): bool
    {
        $current = $this->currentState($order);
        if (! $force && ! $this->can($order, $to)) {
            return false;
        }
        if ($current === $to) {
            return false; // No-op
        }
        $order->state = $to->value;
        $order->status = $to->toDisplayStatus();
        $order->save();
        OrderStateChanged::dispatch($order, $current, $to);
        return true;
    }

    public function currentState(Order $order): OrderState
    {
        if ($order->state) {
            return OrderState::from($order->state);
        }
        // Fallback for legacy rows missing state
        return match($order->status) {
            'In Kitchen' => OrderState::PREPARING,
            'Ready' => OrderState::READY,
            'Out for Delivery' => OrderState::DELIVERING,
            'Delivered' => OrderState::COMPLETED,
            'Cancelled' => OrderState::CANCELLED,
            default => OrderState::AWAITING_KITCHEN,
        };
    }
}
