--
-- Database schema and seed data for Bonny Delight ordering system
--

-- Create database and switch to it
CREATE DATABASE IF NOT EXISTS bonny_delight;
USE bonny_delight;

-- Users table stores customers and staff. Use password_hash() for security.
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  -- Keep full name for backward compatibility with existing pages
  name VARCHAR(200) NOT NULL,
  -- New structured fields
  first_name VARCHAR(100) NOT NULL,
  last_name VARCHAR(100) NOT NULL,
  gender ENUM('male','female') NOT NULL,
  email VARCHAR(100) NOT NULL UNIQUE,
  phone VARCHAR(20) NOT NULL,
  password VARCHAR(255) NOT NULL,
  is_guest BOOLEAN DEFAULT 0,
  room_number VARCHAR(10),
  role ENUM('customer','admin','kitchen','rider') DEFAULT 'customer',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Categories represent menu sections (e.g. Soups, Rice Dishes, etc.)
CREATE TABLE IF NOT EXISTS categories (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL
);

-- Dishes belong to categories and carry price and dietary flags
CREATE TABLE IF NOT EXISTS dishes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  category_id INT,
  name VARCHAR(100) NOT NULL,
  description TEXT,
  price DECIMAL(10,2) NOT NULL,
  image VARCHAR(255),
  vegetarian BOOLEAN DEFAULT 0,
  halal BOOLEAN DEFAULT 1,
  gluten_free BOOLEAN DEFAULT 0,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_dish_category FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
);

-- Orders record purchases made by users. Delivery information and totals
-- are stored here; status tracks fulfilment progress.
CREATE TABLE IF NOT EXISTS orders (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT,
  subtotal DECIMAL(10,2),
  delivery_fee DECIMAL(10,2),
  discount DECIMAL(10,2),
  total DECIMAL(10,2),
  status ENUM('Received','In Kitchen','Ready','Out for Delivery','Delivered') DEFAULT 'Received',
  delivery_address VARCHAR(255),
  payment_method ENUM('Cash on Delivery','Paystack','Charge to Room') DEFAULT 'Cash on Delivery',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_order_user FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Order items link orders to dishes with quantities and item prices
CREATE TABLE IF NOT EXISTS order_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_id INT,
  dish_id INT,
  quantity INT,
  price DECIMAL(10,2),
  CONSTRAINT fk_item_order FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
  CONSTRAINT fk_item_dish FOREIGN KEY (dish_id) REFERENCES dishes(id)
);

-- Seed admin user (email: admin@bonnydelight.ng, password: admin123)
INSERT INTO users (name, first_name, last_name, gender, email, phone, password, role) VALUES
('Admin User','Admin','User','male','admin@bonnydelight.ng','08000000000','0192023a7bbd73250516f069df18b500','admin');

-- Seed categories
INSERT INTO categories (name) VALUES
('Soups & Swallows'),
('Rice Dishes'),
('Grills & Proteins'),
('Stews'),
('Snacks'),
('Desserts'),
('Drinks & Bar'),
('Breakfast'),
('Kids Menu');

-- Seed a few dishes
INSERT INTO dishes (category_id, name, description, price, image, vegetarian, halal, gluten_free) VALUES
(1,'Egusi Soup with Pounded Yam','Rich melon seed soup cooked with leafy vegetables and assorted meats, served with freshly pounded yam.',4000,'assets/images/egusi.jpg',0,1,0),
(2,'Jollof Rice with Grilled Chicken','Classic smoky Nigerian jollof rice served with succulent grilled chicken and side salad.',3500,'assets/images/jollof.jpg',0,1,1),
(1,'Pepper Soup','Light broth spiced with West African seasonings and choice of fish or goat meat.',2500,'assets/images/pepper_soup.jpg',0,1,1),
(4,'Efo Riro','Spinach stew cooked with assorted meats, locust beans and spices.',3000,'assets/images/efo_riro.jpg',0,1,1),
(3,'Suya Platter','Street-style spicy grilled beef skewers garnished with onions, tomatoes and special suya spice.',4500,'assets/images/suya.jpg',0,1,1),
(8,'Akara & Pap','Golden bean fritters served with sweet corn pap for a hearty breakfast.',1500,'assets/images/akara_pap.jpg',1,1,1);

-- Migration assistance for existing installations (MySQL 8+)
-- Add new columns if the users table already exists without them.
ALTER TABLE users ADD COLUMN IF NOT EXISTS first_name VARCHAR(100) NULL AFTER name;
ALTER TABLE users ADD COLUMN IF NOT EXISTS last_name VARCHAR(100) NULL AFTER first_name;
ALTER TABLE users ADD COLUMN IF NOT EXISTS gender ENUM('male','female') NULL AFTER last_name;